# -*- coding: utf-8 -*-
import os


def get_audio_lang(stream_info):
    """
    Example:
        > stream_info = {
                    "index": 0,
                    "codec_name": "aac",
                    ...
                    "tags": {
                        "language": "eng"
                        ...
                    }
        > get_audio_lang(stream_info)
        > eng

    :param stream_info: dict, ffmpeg single stream information
    :return: str, language of audio stream (or und, if no language info available)
    """
    if 'tags' in stream_info and 'language' in stream_info['tags']:
        return stream_info['tags']['language'].lower()
    return None


def get_relative_path(path, pattern='packaged/'):
    return path.split(pattern)[-1]


def get_fairplay_keys(fairplay_output_info, product):
    product_info = filter(
        lambda x: 'product' in x and x['product'] == product,
        fairplay_output_info
    )
    return [
        {k: v for (k, v) in info.items() if k in ('kid, type, product')}
        for info in product_info
    ]

def get_absolute_path(file_name, sandbox_path):
    sb_packaged_path = os.path.join(sandbox_path, "packaged", file_name)
    sb_path = os.path.join(sandbox_path, file_name)
    if os.path.exists(sb_packaged_path):
        return sb_packaged_path
    elif os.path.exists(sb_path):
        return sb_path
    raise ValueError('File {} was not found'.format(file_name))


def file_ends_with(file_name, extensions):
    real_extension = os.path.splitext(file_name)[-1].split('.')[-1].lower()
    if type(extensions) == str:
        return real_extension == extensions
    elif extensions is None:
        return real_extension == ''
    else:
        return real_extension in extensions


def get_s3_path(
        s3_dir,
        s3_sub_dir,
        drm_package_path,
        name,
        drm_package,
        mss_package,
        ott_trailer
        ):
    s3_path = s3_dir + name
    s3_packaged_path = s3_dir + drm_package_path + name

    if ott_trailer:
        # OTT-3793: trailers
        return os.path.join(s3_dir, s3_sub_dir, name)
    elif file_ends_with(name, ('m3u8', 'ts')):
        # VH-4490, OTT-1908: fairplay files
        return s3_path
    elif file_ends_with(name, 'jpg'):
        # OTT-2707: screenshots
        return s3_path
    elif name.endswith('p.mp4'):
        # OTT-4001, VH-6445: original files (169_360p.mp4)
        return s3_path
    elif name.startswith('subtitles/'):
        # OTT-2707: dash and mss subtitles
        return s3_packaged_path
    elif file_ends_with(name, 'vtt'):
        # OTT-2707: hls subtitles
        return s3_path
    elif drm_package or mss_package:
        # mss/dash packaging
        return s3_packaged_path
    else:
        return s3_path


# XXX
def parse_rfc6381_codec(raw_codec):
    codec = raw_codec.lower()
    if codec.startswith('mp4a.'):  # described in rfc6381
        return 'AAC'
    elif codec.startswith('aac'):  # from mss documentation https://nda.ya.ru/3Vqqzd
        return 'AAC'
    elif codec.startswith('ec-3'):  # from ffmpeg source code https://nda.ya.ru/3VogAS
        return 'EC3'
    elif codec.startswith('eac3'):  # from ffmpeg source code https://nda.ya.ru/3VogAV
        return 'EC3'
    elif codec.startswith('avc'):   # described in rfc6381
        return 'AVC'
    elif codec.startswith('h264'):  # from mss documentation https://nda.ya.ru/3Vqqzd
        return 'AVC'
    else:
        raise ValueError('Unknown rfc6381 codec type ' + codec)
