from collections import defaultdict
import re

import requests
from requests.adapters import HTTPAdapter


def get_cache_zones(
    cache_zones_url='http://pinger.strm.yandex.net/api/v1/status/cache',
    cache_zones_url_timeout=3.0,
    cache_zones_url_max_retries=3,
):
    """Get cache zones settings from pinger.

    Example of the response:
    {
        "ams": {
            "strm-ams01.strm.yandex.net": {
                "ads": {
                    "size": 42949672960
                },
                "four": {
                    "size": 1073741824
                },
                ...,
            },
            ...,
        },
        ...,
    }
    """
    session = requests.Session()
    session.mount(
        cache_zones_url,
        HTTPAdapter(
            max_retries=cache_zones_url_max_retries,
        ),
    )

    response = requests.get(
        cache_zones_url,
        timeout=cache_zones_url_timeout,
    )
    response.raise_for_status()
    return response.json()


CONDUCTOR_GROUP_KEY = 'conductor_group'
CONDUCTOR_GROUP_RE = '(?P<{conductor_group}>[a-z]+)'.format(
    conductor_group=CONDUCTOR_GROUP_KEY,
)
RE_HOST_CONDUCTOR_GROUP = re.compile(
    'stor-{re_conductor_group}[0-9]+.strm.yandex.net'.format(
        re_conductor_group=CONDUCTOR_GROUP_RE,
    )
)


def get_conductor_group(host):
    match = RE_HOST_CONDUCTOR_GROUP.match(host)
    if match is None:
        return None

    return match.group(CONDUCTOR_GROUP_KEY)


def get_locations_vod_cache_sizes(
    cache_zones_url='http://pinger.strm.yandex.net/api/v1/status/cache',
    cache_zones_url_timeout=3.0,
    cache_zones_url_max_retries=3,
):
    cache_zones = get_cache_zones(
        cache_zones_url=cache_zones_url,
        cache_zones_url_timeout=cache_zones_url_timeout,
        cache_zones_url_max_retries=cache_zones_url_max_retries,
    )
    cache_zones_informations = [
        cache_zones,
    ]
    processed_hosts = set()
    locations_vod_cache_sizes = defaultdict(int)
    for cache_zones_information in cache_zones_informations:
        for zone, host_informations in cache_zones_information.iteritems():
            for host, cache_zones in host_informations.iteritems():
                if cache_zones is None:
                    continue

                if host in processed_hosts:
                    continue

                processed_hosts.add(host)

                locations_vod_cache_sizes[zone] += cache_zones['vod']['size']

    return locations_vod_cache_sizes
