from collections import namedtuple
import logging

from sandbox.projects.Strm.common.resolution import Resolution
from sandbox.projects.Strm.common.timeout import timeout


logger = logging.getLogger(__name__)


KILOBIT = 1024


class Bitrate(namedtuple('Bitrate', ['value'])):
    """ bitrate is measured in bit/s """
    @property
    def kilo(self):
        return self.value / KILOBIT


FilterOptions = namedtuple('FilterOptions', [
    'resolution',
    'framerate',
    'pixel_format',
    'video_tracks_number',
    'audio_tracks_number',
])


class EncoderOptions(namedtuple('EncoderOptions', [
    'map_metadata',
    'max_muxing_queue_size',
    'stream_duration',
    'video_codec',
    'x264_options',
    'preset',
    'video_profile',
    'level',
    'video_bitrate',
    'video_max_bitrate',
    'video_buffer_size',
    'gop_size',
    'force_key_frames',
    'forced_idr',
    'audio_codec',
    'sample_rate',
    'audio_bitrate',
    'output_format',
])):

    _FLAGS_MAPPING = {
        'forced_idr': 'forced-idr',
        'video_profile': 'profile:v',
        'profile': 'profile:v',
        'preset': 'preset',
        'level': 'level',
        'video_codec': 'c:v',
        'audio_codec': 'c:a',
        'x264_options': 'x264opts',
        'gop_size': 'g',
        'video_max_bitrate': 'maxrate',
        'video_bitrate': 'b:v',
        'audio_bitrate': 'b:a',
        'max_bitrate': 'maxrate',
        'bitrate': 'b:v',
        'buffer_size': 'bufsize',
        'video_buffer_size': 'bufsize',
        'stream_duration': 'to',
        'output_format': 'format',
    }

    def get_ffmpeg_readable(self):
        ffmpeg_readable_options = {}
        for field_name in self._fields:
            value = getattr(self, field_name)
            if value is None:
                continue
            key = field_name
            if EncoderOptions._FLAGS_MAPPING.get(field_name) is not None:
                key = EncoderOptions._FLAGS_MAPPING[field_name]
            ffmpeg_readable_options[key] = value

        if ffmpeg_readable_options.get('b:v') is not None:
            ffmpeg_readable_options['b:v'] = '{}k'.format(ffmpeg_readable_options['b:v'])

        if ffmpeg_readable_options.get('maxrate') is not None:
            ffmpeg_readable_options['maxrate'] = '{}k'.format(ffmpeg_readable_options['maxrate'])

        if ffmpeg_readable_options.get('bufsize') is not None:
            ffmpeg_readable_options['bufsize'] = '{}k'.format(ffmpeg_readable_options['bufsize'])

        if ffmpeg_readable_options.get('b:a') is not None:
            ffmpeg_readable_options['b:a'] = '{}k'.format(ffmpeg_readable_options['b:a'])

        return ffmpeg_readable_options


TranscoderOptions = namedtuple('TranscoderOptions', ['filter', 'encoder'])


VideoTrackInfo = namedtuple('VideoTrackInfo', [
    'resolution',
    'aspect_ratio',
    'codec',
    'sps',
    'pps',
    'bitrate',
    'framerate',
])


AudioTrackInfo = namedtuple('AudioTrackInfo', [
    'codecs',
    'sample_rate',
    'signal_type',
])


StreamInfo = namedtuple('StreamInfo', ['video_tracks', 'audio_tracks'])


RawVideo = namedtuple('RawVideo', [
    'path',
    'resolution',
    'pixel_format',
    'framerate',
    'video',
])


class FFmpeg:

    _AUDIO_CODEC_MAPPING = {'aac': '40'}
    _AUDIO_PROFILE_MAPPING = {'LC': '2'}
    _MP4_AUDIO_CODECS_TPL = 'mp4a.{codec_info}.{profile_info}'

    def __init__(self, ffmpeg_path, ffprobe_path):
        import ffmpeg
        self._ffmpeg = ffmpeg
        self._ffmpeg_path = ffmpeg_path
        self._ffprobe_path = ffprobe_path

    def _run(self, output, timeout_duration=None, overwrite_output=True, capture_stderr=True):
        with timeout(timeout_duration):
            self._ffmpeg.run(output, cmd=self._ffmpeg_path,
                             overwrite_output=overwrite_output, capture_stderr=capture_stderr)

    def _get_raw_video(self, path, resolution, pixel_format, framerate):
        video = self._ffmpeg.input(path, s=resolution.name, pix_fmt=pixel_format, framerate=framerate)
        return RawVideo(
            path=path,
            resolution=resolution,
            pixel_format=pixel_format,
            framerate=framerate,
            video=video,
        )

    def copy_video(self, src_address, dst_address, duration, video_format, timeout_duration=None):
        input_video = self._ffmpeg.input(src_address).video
        output_video = self._ffmpeg.output(input_video, dst_address, vcodec='copy', t=duration, f=video_format)

        command_line = ' '.join(self._ffmpeg.compile(output_video, cmd=self._ffmpeg_path))
        logger.info('Started copying video, command line: {}'.format(command_line))
        try:
            self._run(output_video, timeout_duration)
        except Exception as e:
            reason = e.stderr if isinstance(e, self._ffmpeg.Error) else e.message
            raise RuntimeError('Error copying video: {reason}'.format(reason=reason))

        logger.info('Copied video successfully')
        return

    def encode_video(self, raw_video, dst_address, scaling_resolution, options, timeout_duration=None):
        ffmpeg_readable_options = options.get_ffmpeg_readable()

        input_video = raw_video.video
        scaled_video = self._ffmpeg.filter(input_video, 'scale', scaling_resolution.width, scaling_resolution.height)
        output_video = self._ffmpeg.output(scaled_video, dst_address, **ffmpeg_readable_options)

        command_line = ' '.join(self._ffmpeg.compile(output_video, cmd=self._ffmpeg_path))
        logger.info('Started encoding video, command line: {}'.format(command_line))
        try:
            self._run(output_video, timeout_duration)
        except Exception as e:
            reason = e.stderr if isinstance(e, self._ffmpeg.Error) else e.message
            raise RuntimeError('Error encoding video: {reason}'.format(reason=reason))

        logger.info('Encoded video successfully')
        return

    def decode_video(self, src_address, dst_address, scaling_resolution=None, pix_fmt='yuv420p', deinterlace=False, framerate=None, timeout_duration=None):
        input_video = self._ffmpeg.input(src_address).video

        if deinterlace:
            input_video = self._ffmpeg.filter(input_video, 'yadif', deint=1)

        dst_framerate = self.get_video_info(src_address).video_tracks[0].framerate
        if framerate is not None:
            input_video = self._ffmpeg.filter(input_video, 'fps', fps=framerate)
            dst_framerate = framerate

        dst_resolution = self.get_video_info(src_address).video_tracks[0].resolution
        if scaling_resolution is not None:
            input_video = self._ffmpeg.filter(input_video, 'scale', scaling_resolution.width, scaling_resolution.height)
            dst_resolution = scaling_resolution

        output_video = self._ffmpeg.output(input_video, dst_address, vcodec='rawvideo', pix_fmt=pix_fmt)

        command_line = ' '.join(self._ffmpeg.compile(output_video, cmd=self._ffmpeg_path))
        logger.info('Started decoding video, command line: {}'.format(command_line))
        try:
            self._run(output_video, timeout_duration)
        except Exception as e:
            reason = e.stderr if isinstance(e, self._ffmpeg.Error) else e.message
            raise RuntimeError('Error decoding video: {reason}'.format(reason=reason))

        logger.info('Decoded video successfully')

        raw_video = self._get_raw_video(
            path=dst_address,
            resolution=dst_resolution,
            framerate=dst_framerate,
            pixel_format=pix_fmt,
        )
        return raw_video

    @staticmethod
    def _get_sps_pps(video_path):
        from contrib.tools.mp4viewer.src import datasource, showboxes
        with open(video_path, 'rb') as fd:
            boxes = showboxes.getboxlist(datasource.DataBuffer(datasource.FileSource(fd)))
        # moov box should exist, see ISO/IEC 14496-12
        for box in boxes:
            if box.boxtype == 'moov':
                moov_box = box
                break
        sps_pps = {}
        for box in moov_box.children:
            if box.boxtype != 'trak':
                continue

            mdia_box = box.find_child('mdia')
            minf_box = mdia_box.find_child('minf')
            # vmhd box is exists only for video tracks
            if minf_box.find_child('vmhd') is None:
                continue
            stbl_box = minf_box.find_child('stbl')
            stsd_box = stbl_box.find_child('stsd')

            avc1_box = stsd_box.find_child('avc1')
            if avc1_box is None:
                continue
            avc1_box = stsd_box.find_child('avc1')
            avcC_box = avc1_box.find_child('avcC')
            for field in avcC_box.generate_fields():
                if field[0] != 'SPS' and field[0] != 'PPS':
                    continue
                sps_pps[field[0]] = field[1]

        return sps_pps

    @staticmethod
    def _get_audio_codecs(codec_name, profile):
        return FFmpeg._MP4_AUDIO_CODECS_TPL.format(
            codec_info=FFmpeg._AUDIO_CODEC_MAPPING[codec_name],
            profile_info=FFmpeg._AUDIO_PROFILE_MAPPING[profile],
        )

    # now works correctly only for streams with not more than one video_track
    def get_video_info(self, stream_path):
        try:
            probe = self._ffmpeg.probe(stream_path, cmd=self._ffprobe_path)
        except self._ffmpeg.Error as e:
            raise RuntimeError('Failed to get info from video source: {reason}'.format(reason=e.stderr))

        audio_tracks_info = []
        video_tracks_info = []
        for track_info in probe['streams']:
            if track_info['codec_type'] == 'video':
                width = int(track_info['width'])
                height = int(track_info['height'])
                resolution = Resolution(width, height)
                aspect_ratio = resolution.get_aspect_ratio()
                sps_pps = FFmpeg._get_sps_pps(stream_path)
                # r_frame_rate field is in format 'numerator/denominator'
                framerate_ratio = track_info['r_frame_rate'].split('/')
                framerate = float(framerate_ratio[0]) / float(framerate_ratio[1])
                video_track_info = VideoTrackInfo(
                    resolution=resolution,
                    aspect_ratio=aspect_ratio,
                    codec=str(track_info['codec_name']),
                    sps=sps_pps['SPS'],
                    pps=sps_pps['PPS'],
                    bitrate=Bitrate(int(track_info['bit_rate'])).kilo,
                    framerate=framerate,
                )
                video_tracks_info.append(video_track_info)
            elif track_info['codec_type'] == 'audio':
                audio_codecs = FFmpeg._get_audio_codecs(track_info['codec_name'], track_info['profile'])
                audio_track_info = AudioTrackInfo(
                    codecs=audio_codecs,
                    sample_rate=int(track_info['sample_rate']),
                    signal_type=str(track_info['channel_layout']),
                )
                audio_tracks_info.append(audio_track_info)

        return StreamInfo(
            video_tracks=video_tracks_info,
            audio_tracks=audio_tracks_info,
        )

    def _fit_to_resolution(self, video_stream, resolution):
        # DAR is display aspect ratio
        # SAR is sample aspect ratio
        # View https://en.wikipedia.org/wiki/Pixel_aspect_ratio for details
        # Fit frames to rectangle of size (resolution.width / SAR, resolution.height)
        pre_scaled = self._ffmpeg.filter(
            video_stream,
            'scale',
            w='trunc(min({width},{height}*dar)/sar/2)*2'.format(
                width=resolution.width,
                height=resolution.height
            ),
            h='-1',
        )
        # Add black padding if DAR of original resolution is not equal to DAR of new resolution
        padded = self._ffmpeg.filter(
            pre_scaled,
            'pad',
            w='ceil({width}/sar/2)*2'.format(width=resolution.width),
            h=resolution.height,
            x='(ow-iw)/2',
            y='(oh-ih)/2',
            color='black',
        )
        # Scale as if SAR is 1/1
        scaled = self._ffmpeg.filter(
            padded,
            'scale',
            in_range='auto',
            out_range='pc',
            w=resolution.width,
            h=resolution.height,
        )
        # Set SAR to 1/1
        return self._ffmpeg.filter(scaled, 'setsar', '1/1')

    def _filter(self, stream, options):
        audio_track = stream.audio
        output_tracks = []
        # Add audio_tracks_number copies of input audio to output
        output_tracks.extend([audio_track] * options.audio_tracks_number)

        video_track = stream.video
        if options.framerate is not None:
            # Change fps of input video
            video_track = self._ffmpeg.filter(video_track, 'fps', options.framerate)

        if options.pixel_format is not None:
            # Change pixel format of input video
            video_track = self._ffmpeg.filter(video_track, 'format', options.pixel_format)

        if options.resolution is not None:
            # Fit video to particular resolution
            video_track = self._fit_to_resolution(video_track, options.resolution)

        # Add video_tracks_number copies of input video to output
        output_tracks.extend([video_track] * options.video_tracks_number)

        return output_tracks

    def transcode_video(self, src_address, dst_address, options, timeout_duration=None):
        input_stream = self._ffmpeg.input(src_address)
        output_tracks = self._filter(input_stream, options.filter)
        streams_and_filename = output_tracks + [dst_address]
        output = self._ffmpeg.output(*streams_and_filename, **options.encoder.get_ffmpeg_readable())

        command_line = ' '.join(self._ffmpeg.compile(output, cmd=self._ffmpeg_path))
        logger.info('Started transcoding video, command line: {}'.format(command_line))
        try:
            self._run(output, timeout_duration)
        except Exception as e:
            reason = e.stderr if isinstance(e, self._ffmpeg.Error) else e.message
            raise RuntimeError('Error transcoding video: {reason}'.format(reason=reason))

        logger.info('Transcoded video successfully')
        return
