from collections import namedtuple
import requests


ChannelConfig = namedtuple('ChannelConfig', [
    'src_groups_names',
    'port',
])


class Pult:
    _CHANNEL_URL_TPL = '{url}/{channel_name}'

    def __init__(self, url):
        self._url = url
        self._config = {}

    def update(self):
        try:
            response = requests.get(url=self._url, verify=False)
        except Exception as e:
            raise RuntimeError('Unable to get response: {reason}'.format(reason=e.message))

        try:
            config = response.json()
        except ValueError as e:
            raise ValueError('Unable to parse response: {reason}'.format(reason=e.message))

        self._config = config['sources']
        return

    def update_channel(self, channel_name):
        request_url = self._CHANNEL_URL_TPL.format(url=self._url, channel_name=channel_name)

        try:
            response = requests.get(url=request_url, verify=False)
        except Exception as e:
            raise RuntimeError('Unable to get response: {reason}'.format(reason=e.message))

        try:
            self._config[channel_name] = response.json()
        except ValueError as e:
            raise ValueError('Unable to parse response: {reason}'.format(reason=e.message))

        return

    def get_channel_cfg(self, channel_name, full_update=False, channel_update=False):
        if full_update:
            try:
                self.update()
            except Exception as e:
                raise RuntimeError('Failed to update pult config: {reason}'.format(reason=e.message))

        if channel_update:
            try:
                self.update_channel(channel_name)
            except Exception as e:
                raise RuntimeError('Failed to update pult config for channel {channel_name}: {reason}'.format(
                    channel_name=channel_name,
                    reason=e.message
                ))

        if not self._config:
            try:
                self.update()
            except Exception as e:
                raise RuntimeError('Failed to update pult config: {reason}'.format(reason=e.message))

        channel_config = self._config.get(channel_name)
        if channel_config is None:
            raise RuntimeError('Failed to get pult config for channel {channel_name}: {reason}'.format(
                channel_name=channel_name,
                reason=e.message
            ))

        src_groups = channel_config['src_groups']
        if not src_groups:
            raise RuntimeError('No src_groups')

        src_groups_names = src_groups.iterkeys()

        port = channel_config['src_port']

        return ChannelConfig(src_groups_names, port)
