from collections import namedtuple


class Resolution(namedtuple('Resolution', ['width', 'height'])):

    _RESOLUTION_ASPECT_RATIO_MAPPING = {
        '426x240': '169',
        '432x240': '169',
        '480x270': '169',
        '640x360': '169',
        '852x480': '169',
        '1024x576': '169',
        '1280x720': '169',
        '1920x1080': '169',
        '2560x1440': '169',
        '3840x2160': '169',

        '320x240': '43',
        '360x270': '43',
        '480x360': '43',
        '640x480': '43',
        '720x540': '43',
        '768x576': '43',
        '960x720': '43',
        '1440x1080': '43',

        '436x240': '2011',
        '654x360': '2011',
        '872x480': '2011',
        '1046x576': '2011',
        '1310x720': '2011',

        '326x240': '1511',
        '490x360': '1511',
        '654x480': '1511',
        '784x576': '1511',
        '980x720': '1511',
        '1472x1080': '1511',

        '300x240': '54',
        '450x360': '54',
        '600x480': '54',
        '720x576': '54',
        '900x720': '54',
        '1350x1080': '54',
    }
    _QUALITY_TPL = '{aspect_ratio}_{height}p'

    @property
    def name(self):
        return '{}x{}'.format(self.width, self.height)

    @property
    def scale(self):
        return '{}:{}'.format(self.width, self.height)

    @staticmethod
    def parse_from_name(name):
        try:
            width, height = map(int, name.split('x'))
        except Exception as e:
            raise RuntimeError('Could not parse resolution name: {reason}'.format(reason=e.message))

        return Resolution(width, height)

    def get_aspect_ratio(self):
        return Resolution._RESOLUTION_ASPECT_RATIO_MAPPING[self.name]

    def get_quality(self):
        return Resolution._QUALITY_TPL.format(
            aspect_ratio=self.get_aspect_ratio(),
            height=self.height,
        )
