import json
import logging
import os

from sandbox import sdk2
import sandbox.common.types.resource as resource
from sandbox.projects import resource_types

from sandbox.projects.Strm.common.ffmpeg_wrapper import FFmpeg


logger = logging.getLogger(__name__)


def _download_resource(**resource_filters):
    try:
        resources = list(sdk2.Resource.find(state=[resource.State.READY], **resource_filters).limit(1))
    except Exception as e:
        raise RuntimeError('Failed to find resource: {reason}'.format(reason=e.message))

    if not resources:
        raise RuntimeError('No resources found')

    found_resource = resources[0]

    try:
        resource_data = sdk2.ResourceData(found_resource)
    except Exception as e:
        raise RuntimeError('Failed to get resource data: {reason}'.format(reason=e.message))

    logger.info('Resource downloaded: %s', found_resource.description)
    return resource_data


def get_ffmpeg():
    try:
        ffmpeg_resource = _download_resource(resource_type=resource_types.FFMPEG_BIN)
    except Exception as e:
        raise RuntimeError('Could not download {name} resource: {reason}'.format(name=resource_types.FFMPEG_BIN.__name__, reason=e.message))
    ffmpeg_path = os.path.join(str(ffmpeg_resource.path), 'ffmpeg')
    ffprobe_path = os.path.join(str(ffmpeg_resource.path), 'ffprobe')
    return FFmpeg(ffmpeg_path, ffprobe_path)


def get_vmaf_credentials():
    try:
        vmaf_resource = _download_resource(id=1454487735)
    except Exception as e:
        raise RuntimeError('Could not download vmaf resource: {reason}'.format(reason=e.message))
    vmaf_credentials = {
        'path': os.path.join(str(vmaf_resource.path), 'usr/bin/vmaf'),
        'model_path': os.path.join(str(vmaf_resource.path), 'usr/share/vmaf/model/vmaf_v0.6.1.pkl'),
    }
    return vmaf_credentials


def load_json_resource(**resource_filters):
    try:
        json_resource = _download_resource(**resource_filters)
    except Exception as e:
        raise RuntimeError('Failed to get json resource: {reason}'.format(reason=e.message))

    try:
        with open(str(json_resource.path), 'r') as json_file:
            data = json.load(json_file)
    except Exception as e:
        raise RuntimeError('Failed to parse json data from resource: {reason}'.format(reason=e.message))

    return data
