import boto3
from boto3.s3.transfer import S3Transfer
import datetime
import json
import pytz

from sandbox.projects.Strm.common.temporary_file import create_tempfile


class S3Client:

    def __init__(self, url, secret_key_id, secret_key):
        self._client = boto3.client(
            's3',
            endpoint_url=url,
            aws_access_key_id=secret_key_id,
            aws_secret_access_key=secret_key,
        )

        self._transfer = S3Transfer(self._client)

    def upload(self, s3_bucket, local_path, s3_path):
        try:
            self._transfer.upload_file(local_path, s3_bucket, s3_path)
        except Exception as e:
            raise RuntimeError('Failed to upload to S3: {reason}'.format(reason=e.message))

        return

    def download(self, s3_bucket, local_path, s3_path):
        try:
            self._transfer.download_file(s3_bucket, s3_path, local_path)
        except Exception as e:
            raise RuntimeError('Failed to download from S3: {reason}'.format(reason=e.message))

        return

    def delete(self, bucket, paths):
        try:
            self._client.delete_objects(
                Bucket=bucket,
                Delete={
                    'Objects': [
                        {'Key': path}
                        for path in paths
                    ]
                }
            )
        except Exception as e:
            raise RuntimeError('Failed to delete from S3: {reason}'.format(reason=e.message))
        return

    def list_files(self, s3_bucket, prefix, max_file_age):
        resp = self._client.list_objects_v2(Bucket=s3_bucket, Prefix=prefix)
        files_list = []
        for obj in resp.get('Contents', []):
            date_modified = obj['LastModified']
            if datetime.datetime.now(pytz.utc) - date_modified <= max_file_age:
                files_list.append(obj['Key'])
        return files_list

    def delete_outdated(self, bucket, prefix, max_file_age):
        resp = self._client.list_objects_v2(Bucket=bucket, Prefix=prefix)
        delete_list = []
        for obj in resp.get('Contents', []):
            date_modified = obj['LastModified']
            if datetime.datetime.now(pytz.utc) - date_modified > max_file_age:
                delete_list.append(obj['Key'])
        try:
            self.delete(bucket, delete_list)
        except Exception as e:
            raise RuntimeError('Failed to delete outdated files with prefix {prefix} from bucket {bucket} in S3: {reason}'.format(
                bucket=bucket,
                prefix=prefix,
                reason=e.message,
            ))
        return

    def json_load(self, bucket, s3_json_path):
        with create_tempfile() as tmp_file_path:
            try:
                self.download(bucket, tmp_file_path, s3_json_path)
            except Exception as e:
                raise RuntimeError('Failed to download json from S3: {reason}'.format(reason=e.message))

            with open(tmp_file_path, 'r') as tmp_file:
                try:
                    load = json.load(tmp_file)
                except Exception as e:
                    raise RuntimeError('Could not parse json: {reason}'.format(reason=e.message))
                return load

    def json_dump(self, bucket, data, s3_json_path):
        with create_tempfile() as tmp_file_path:
            with open(tmp_file_path, 'w') as tmp_file:
                json.dump(data, tmp_file, indent=4, sort_keys=True)
            try:
                self.upload(bucket, tmp_file_path, s3_json_path)
            except Exception as e:
                raise RuntimeError('Failed to upload json to S3: {reason}'.format(reason=e.message))


def get_s3_client(url, credentials_owner, vault_key='S3_CREDENTIALS'):
    from sandbox import sdk2
    try:
        credentials = sdk2.Vault.data(credentials_owner, vault_key)
    except Exception as e:
        raise RuntimeError('Failed to get S3 credentials: {reason}'.format(reason=e.message))

    import contrib.python.configparser as configparser
    config = configparser.ConfigParser()
    config.read_string(unicode(credentials))
    secret_key = config['Credentials']['aws_secret_access_key']
    secret_key_id = config['Credentials']['aws_access_key_id']

    return S3Client(url, secret_key_id, secret_key)
