import io
import json
import logging


logger = logging.getLogger(__name__)


def execute_yql_query(client, yql_query, syntax_version=0, attach_files=None, title=None, parameters=None):
    # type: (YqlClient, str, int, list[tuple[str, str]], str, dict[str, str]) -> YqlTable

    if title is not None and 'YQL' not in title:
        title += ' YQL'  # to avoid 'Please mention YQL in the title' exception

    request = client.query(
        yql_query,
        syntax_version=syntax_version,
        title=title,
    )
    if attach_files:
        for (name, data) in attach_files:
            request.attach_file_data(data=data, name=name)

    request.run(parameters=parameters)

    if not request.get_results().is_success:
        error_description = '\n'.join(
            str(err)
            for err in request.get_results().errors
        )
        raise RuntimeError('Execution of yql query failed: %s' % error_description)

    table = request.get_results().table
    table.fetch_full_data()
    return table


def dump_yql_table_to_resource(yql_table, result_resource):
    result_file_name = str(result_resource.path)
    logger.info('Dump results to file: %s', result_file_name)

    rows = yql_table.rows
    keys = [key for key, _ in yql_table.columns]
    result = [
        dict(zip(keys, row))
        for row in rows
    ]

    with io.open(result_file_name, 'w', encoding='utf-8') as res_file:
        res_file.write(
            json.dumps(
                result,
                ensure_ascii=False,
                encoding='utf-8',
            ),
        )
