import os
import six
import shlex
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers import subprocess as sp

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

import sandbox.projects.sandbox.resources as sb_resources
from sandbox.projects.sandbox import sandbox_lxc_image_acceptance
from sandbox.projects.sandbox import remote_copy_resource

from sandbox.projects.common import file_utils as fu

from . import image


UbuntuRelease = image.UbuntuRelease


class RichTextTaskFailure(common.errors.TaskFailure):
    def __init__(self, message, rich_addition):
        super(RichTextTaskFailure, self).__init__(message)
        self.rich_addition = rich_addition
        self.message = message

    def __str__(self):
        return "Error was occured: {}. Info: {}".format(self.message, self.rich_addition)

    def get_task_info(self):
        return self.rich_addition


class SandboxLxdImage(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Install contains 4 stages:
    1. Prepare host system
    2. Prepare basic image (bootstrap with BOOTSTRAP_* vars, dist-upgrade)
    3. Customize basic image (Install search source-list, install chroot extra packages,
       Execute chroot extra commands, create chroot conf files)
    4. Compress image to file `IMAGE_FILE_NAME`
    """

    __bootstrap = None
    __image = None

    IMAGE_SIZE_LIMIT = 15 * 1024

    @property
    def rootfs(self):
        return str(self.ramdrive.path / "rootfs")

    CHROOT_COPY_FILES = [
        "/etc/hostname",
        "/etc/resolv.conf",
        "/etc/network/interfaces",
    ]

    _custom_source = {}
    _postcook_commands = []
    _resources = {}

    OWN_LOGGER_NAME = "shell"
    OWN_LOG_FILE_PATH = "shell.log"

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        privileged = True
        dns = ctm.DnsType.DNS64
        disk_space = 16 * 1024
        cores = 1
        ram = 16000

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String(
            "Container resource type", required=True, default=sb_resources.LXC_CONTAINER.name
        ) as resource_type:
            # noinspection PyTypeChecker
            for rt in sdk2.Resource:
                resource_type.values[rt.name] = rt.name

        image_name = sdk2.parameters.String("Image name", default="rootfs.tar.gz")

        resource_description = sdk2.parameters.String("Container resource's custom description")

        with sdk2.parameters.String("Ubuntu release", default=image.UbuntuRelease.BIONIC) as ubuntu_release:
            # noinspection PyTypeChecker
            for release in image.UbuntuRelease:
                ubuntu_release.values[release] = release

        test_result_lxc = sdk2.parameters.Bool("Test resulting container", default=False)

        custom_image = sdk2.parameters.Bool("Create custom LXC image")
        with custom_image.value[True]:
            with sdk2.parameters.Group("Custom image settings") as no_one_reads_this_anyway:
                install_common = sdk2.parameters.Bool(
                    "Install all common packages",
                    description=(
                        "Install all packages (as for regular container) listed in files "
                        "`deb_packages_common` and `deb_packages_specific.{PLATFORM}` in the task's directory."
                    ),
                    default=False
                )
                custom_repos = sdk2.parameters.String(
                    "Additional package repositories",
                    description="Will be added to source.list.d/custom.list",
                    multiline=True
                )

                custom_script = sdk2.parameters.String(
                    "Shell script to execute during final stage",
                    description=(
                        "Will be copied to the container and started with '/bin/bash /custom.sh' after chroot. "
                        "Stdout of script could be found in shell.log"
                    ),
                    default="rm -rf /Berkanavt /opt/skynet",
                    multiline=True
                )

                base_packages = sdk2.parameters.String(
                    "Space-separated list of packages to install in chrooted environment before building the container",
                    default=" ".join(image.Image.BASE_PACKAGES),
                    multiline=True
                )
                custom_packages = sdk2.parameters.String(
                    "List of packages to install during final stage, space-separated",
                    multiline=True
                )
                unwanted_packages = sdk2.parameters.String(
                    "List of packages to purge during final stage, space-separated",
                    multiline=True,
                    default=" ".join(("juggler-client", "^config-juggler", ""))
                )

                resources = sdk2.parameters.Dict(
                    "Resources to add to the container image (key - resource_id/rbtorrent, value - path)",
                    default={}
                )

                custom_env_secrets = sdk2.parameters.Dict(
                    "List of secrets to use in shell script",
                    description=(
                        "Secrets that will be used in your custom_script as environment variables. "
                        "Format: variable_name -> secret_id@version#key ('key' is required here)"
                    )
                )

        with sdk2.parameters.Group("Other settings") as other_settings:
            bin_params = binary_task.LastBinaryReleaseParameters()


    class Context(sdk2.Task.Context):
        children = []
        container_id = 0
        copy_task_id = None

    @property
    def exec_logger(self):
        try:
            return self.__exec_logger
        except AttributeError:
            exec_logger = logging.getLogger(self.OWN_LOGGER_NAME)
            map(exec_logger.removeHandler, exec_logger.handlers[:])
            handler = logging.FileHandler(str(self.log_path(self.OWN_LOG_FILE_PATH)))
            handler.setFormatter(logging.Formatter("%(asctime)s\t%(message)s"))
            exec_logger.addHandler(handler)
            exec_logger.propagate = False
            # noinspection PyAttributeOutsideInit
            self.__exec_logger = exec_logger

        return self.__exec_logger

    def _extract_custom_env_secrets(self):
        logging.info("Extracting secrets to environment variables")
        acceptable_secrets = {}
        for name, secret in self.Parameters.custom_env_secrets.items():
            secret = sdk2.yav.Secret.__decode__(secret)
            if secret.default_key is not None:
                acceptable_secrets[name] = str(secret.value())
            else:
                logging.warning("Can't extract %s without default_key", name)
        return acceptable_secrets

    def execute(self, command_line, chroot=False, add_custom_env_secrets=False):
        """
        Take text of shell command.
        If chroot=True - execute command in chroot env
        """

        if chroot:
            command_line = "chroot " + self.rootfs + " bash -c '" + command_line + "'"
        else:
            command_line = "bash -c '" + command_line + "'"
        args = shlex.split(command_line)
        my_env = os.environ.copy()
        my_env["TMPDIR"] = "/var/tmp"

        if add_custom_env_secrets:
            my_env.update(self._extract_custom_env_secrets())

        self.exec_logger.info("\t>>> EXECUTING COMMAND: %s", command_line)
        with sdk2.helpers.ProcessLog(self, logger=self.exec_logger, set_action=False) as pl:
            try:
                returncode = sp.Popen(args, stdout=pl.stdout, stderr=sp.STDOUT, env=my_env).wait()
                if returncode == 0:
                    return True
                raise RichTextTaskFailure(
                    "Command {!r} failed, see the log below.".format(command_line),
                    "Shell commands output written to <b><a href='{}'>{}</a></b>".format(
                        "/".join((self.log_resource.http_proxy, self.OWN_LOG_FILE_PATH)),
                        self.OWN_LOG_FILE_PATH
                    )
                )
            except Exception:
                logging.exception("SUBPROCESS ERROR")
                self.umount()
                raise

    def prepare(self):
        self.execute("install -d -o root " + self.rootfs)
        self.execute("apt-get update -y")
        self.execute("apt-get install -y " + " ".join(self.__bootstrap.REQUIREMENTS))
        return "Prepared"

    def bootstrap(self):
        """ Create basic system """
        self.execute(self.__bootstrap.cmd(self.rootfs))
        self.execute("mkdir -p " + self.rootfs + "/tmp/metadata/templates")
        return "Bootstrapped"

    def mount(self):
        self.execute("mount --bind /dev/ " + self.rootfs + "/dev/")
        if self.Parameters.ubuntu_release != image.UbuntuRelease.LUCID:
            self.execute("mount --bind /run/ " + self.rootfs + "/run/")
        self.execute("mount --bind /dev/pts/ " + self.rootfs + "/dev/pts/")
        self.execute("mount --bind /tmp/ " + self.rootfs + "/tmp/")

        for path in (
            "/opt/skynet/",  # certain Debian packages require SkyNET to be installed: SANDBOX-5862
        ):
            endpoint = self.rootfs + path
            if not os.path.exists(endpoint):
                os.makedirs(endpoint)
            self.execute("mount --bind {} {} ".format(path, endpoint))

        self.execute("mount -t proc none " + self.rootfs + "/proc/")
        self.execute("mount -t sysfs none " + self.rootfs + "/sys/")
        return "Mounted"

    def umount(self):
        self.execute("umount -l " + self.rootfs + "/dev/pts/ || true")
        self.execute("umount -l " + self.rootfs + "/dev/ || true")
        self.execute("umount -l " + self.rootfs + "/run/ || true")
        self.execute("umount -l " + self.rootfs + "/proc/ || true")
        self.execute("umount -l " + self.rootfs + "/sys/ || true")
        self.execute("umount -l " + self.rootfs + "/tmp/ || true")
        self.execute("umount -l " + self.rootfs + "/opt/skynet/ || true")

        return "Umounted"

    def check_config(self, conf_list):
        """
        Takes dict of {"file": value} and check files
        value may be dict or list
        """
        for path in conf_list.iterkeys():
            if os.path.isfile(self.rootfs + path):
                logging.info("File " + self.rootfs + path + " exists")
            else:
                raise common.errors.TaskFailure("Can't create file " + self.rootfs + path)
        return True

    def add_config(self, conf_list):
        """
        Takes dict of {"file": value} and add it into container
        value may be dict or list
        dict:  {"owner": 0, "group": 0, "mask": 0440, "content": ["line1", "line2"]}
        """
        if not conf_list:
            return "Nothing to create"
        for path, configs in conf_list.iteritems():
            if isinstance(configs, dict):
                owner = configs.get("owner", 0)
                group = configs.get("group", 0)
                mask = configs.get("mask", 0o755)
                content = configs.get("content", [])
            else:
                owner = None
                group = None
                mask = None
                content = configs

            file_path = self.rootfs + path
            logging.info("Trying to add %s", file_path)
            with open(file_path, "w") as f:
                f.writelines(
                    "\n".join(
                        map(lambda _: _.encode("utf8") if isinstance(_, six.text_type) else _, content) + [""]
                    )
                )
            if all(_ is not None for _ in (owner, group, mask)):
                os.chown(file_path, owner, group)
                os.chmod(file_path, mask)

        logging.info("Checking for created files")
        self.check_config(conf_list)

        return "Files created"

    def dist_upgrade(self):
        """
        Do dist-upgrade in base system
        """
        # Add ubuntu sources
        logging.info(self.add_config(self.__image.base_sources))

        # Exec apt-get upgrade
        self.execute(
            "(mkdir /root/sbin && for i in initctl invoke-rc.d restart start stop start-stop-daemon service; "
            "do cp /bin/true /root/sbin/$i; done) || True ; echo true",
            True
        )

        # in some containers, /etc/resolv.conf in rootfs is a symlink to a stub which does not exist
        resolv_conf_path = "/etc/resolv.conf"
        self.execute("rm {}{}".format(self.rootfs, resolv_conf_path))
        self.execute("cp {resolv_conf} {rootfs}{resolv_conf}".format(
            rootfs=self.rootfs, resolv_conf=resolv_conf_path)
        )
        self.execute("DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH apt-get -y update", True)
        self.execute("DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH apt-get -y --force-yes dist-upgrade", True)
        self.execute("DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH apt-get clean ", True)
        self.execute("DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH apt-get autoclean", True)
        self.execute("> " + self.rootfs + "/root/.bash_history")
        return "Rootfs upgraded"

    def cook_image(self, current_dir):
        """
        Customize basic image.
        Add search repos, install additional packages, exec custom cmd
        """

        logging.info(self.add_config(self.__image.specific_conf_files))
        if self._custom_source:
            logging.info(self.add_config(self._custom_source))

        # Download source-lists packages to container
        os.chdir("".join([self.rootfs, "/var/cache/apt/archives/"]))
        self.execute("apt-get --yes download " + " ".join(self.__image.base_packages))
        os.chdir(current_dir)

        # And install source list packages
        self.execute("dpkg -i /var/cache/apt/archives/*.deb", True)

        if self.Parameters.ubuntu_release == image.UbuntuRelease.PRECISE:

            for file_name in fu.find_files("/etc/apt", "*"):
                logging.info("Patching %s", file_name)
                deb_contents = fu.read_file(file_name)

                deb_contents_new = deb_contents.replace('mirror.yandex.ru/ubuntu', 'mirror.yandex.ru/old-ubuntu')
                if deb_contents_new != deb_contents:
                    logging.info("File %s has been patched", file_name)
                    fu.write_file(file_name, deb_contents_new)

        # Install extra packages
        self.execute("DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH apt-get --yes update", True)
        packages = " ".join(self.__image.extra_packages)
        if packages:
            self.execute(
                "DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH apt-get --yes --force-yes install " + packages,
                True
            )

        logging.info("Execute custom commands")
        for command in image.file2list("chroot_extra_cmds.sh"):
            self.execute(command, True)

        # Copy files to rootfs
        for copy_file in self.CHROOT_COPY_FILES:
            self.execute("cp " + copy_file + " " + self.rootfs + copy_file)

        # Empty /etc/hosts by default
        self.add_config({"/etc/hosts": image.file_descr(mask=0o0664)})

        self.add_resources_to_image()

        return "Image cooked"

    def execute_custom_script(self, custom_script):
        logging.info(self.add_config({"/custom.sh": image.file_descr(content=custom_script.splitlines())}))
        self.execute("/bin/bash /custom.sh", True, add_custom_env_secrets=True)
        self.execute("/bin/rm /custom.sh", True)

        return "Custom script finished"

    def postcook(self):
        for command in self._postcook_commands:
            self.execute(command, True)

        for package in self.Parameters.unwanted_packages.split():
            self.execute(
                "DEBIAN_FRONTEND=noninteractive PATH=/root/sbin:$PATH "
                "apt-get --yes --force-yes purge {} || true".format(package)
            )

        logging.info("Execute postcook common commands")
        for command in image.file2list("chroot_postook_cmds.sh"):
            self.execute(command, True)

        logging.info(self.add_config(self.__image.conf_files))
        return "Postcook finished"

    def compress(self):
        image_name = self.Parameters.image_name
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("tar")) as pl:
            cmd0 = [
                "mv", str(self.ramdrive.path / "rootfs" / "tmp" / "metadata" / "templates"), str(self.ramdrive.path)
            ]
            cmd1 = [
                "mv", str(self.ramdrive.path / "rootfs" / "tmp" / "metadata" / "metadata.yaml"), str(self.ramdrive.path)
            ]
            cmd2 = [
                "rm", "-rf", str(self.ramdrive.path / "rootfs" / "tmp" / "metadata")
            ]
            cmd3 = [
                "tar", "caf", image_name, "-C", str(self.ramdrive.path), "rootfs", "templates", "metadata.yaml"
            ]
            sp.check_call(cmd0, stdout=pl.stdout, stderr=pl.stderr)
            sp.check_call(cmd1, stdout=pl.stdout, stderr=pl.stderr)
            sp.check_call(cmd2, stdout=pl.stdout, stderr=pl.stderr)
            sp.check_call(cmd3, stdout=pl.stdout, stderr=pl.stderr)
        logging.info("Image %r created", image_name)
        return image_name

    def create_image(self):
        current_dir = os.path.realpath("")

        if self.Parameters.resources:
            self._resources = self.find_resources()

        self._postcook_commands = image.file2list("postcook_cmds." + self.Parameters.ubuntu_release)

        if self.Parameters.custom_image:
            self._custom_source["/etc/apt/sources.list.d/custom.list"] = self.Parameters.custom_repos.splitlines()

        with sdk2.helpers.ProgressMeter("Preparing..."):
            logging.info(self.prepare())
        with sdk2.helpers.ProgressMeter("Bootstrapping..."):
            logging.info(self.bootstrap())
        with sdk2.helpers.ProgressMeter("Mounting..."):
            logging.info(self.mount())
        with sdk2.helpers.ProgressMeter("Running dist-upgrade..."):
            logging.info(self.dist_upgrade())
        with sdk2.helpers.ProgressMeter("Cooking the image..."):
            logging.info(self.cook_image(current_dir))
            if self.Parameters.custom_image:
                with sdk2.helpers.ProgressMeter("Customizing the image..."):
                    logging.info(self.execute_custom_script(self.Parameters.custom_script))
        with sdk2.helpers.ProgressMeter("Unmounting the image..."):
            logging.info(self.umount())
        with sdk2.helpers.ProgressMeter("Postcooking the image..."):
            logging.info(self.postcook())
        with sdk2.helpers.ProgressMeter("Compressing the image..."):
            image_size = common.fs.get_dir_size(self.rootfs) >> 10
            if image_size > self.IMAGE_SIZE_LIMIT:
                raise common.errors.TaskFailure("Resulting image size {} exceeds maximum of {}".format(
                    common.utils.size2str(image_size << 20), common.utils.size2str(self.IMAGE_SIZE_LIMIT << 20)
                ))
            image_name = self.compress()

        resource_attributes = {
            ctr.ServiceAttributes.TTL: "inf",
            "platform": common.platform.get_platform_alias(self.Parameters.ubuntu_release),
        }

        # noinspection PyArgumentList
        lxc_resource = sdk2.Resource[self.Parameters.resource_type](
            self,
            description=(
                self.Parameters.resource_description.strip() or
                "LXC root FS for {0}".format(self.Parameters.ubuntu_release)
            ),
            path=image_name,
            **resource_attributes
        )

        sdk2.ResourceData(lxc_resource).ready()
        self.Context.container_id = lxc_resource.id

    def run_tests(self):
        for privileged in (True, False):
            task = sandbox_lxc_image_acceptance.SandboxLxcImageAcceptance(
                self,
                description="Test LXC image #{}".format(self.Context.container_id),
                lxc_resource=self.Context.container_id,
                privileged=privileged
            )
            task.Requirements.disk_space = self.Requirements.disk_space
            self.Context.children.append(task.save().enqueue().id)

        raise sdk2.WaitTask(self.Context.children, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))

    def check_children_state(self):
        if not all(
            task.status in ctt.Status.Group.SUCCEED
            for task in self.find(id=self.Context.children)
        ):
            raise common.errors.TaskFailure("The container didn't pass acceptance checks (see subtasks for details)")

    def on_save(self):
        if self.Parameters.ubuntu_release in (image.UbuntuRelease.BIONIC, image.UbuntuRelease.FOCAL):
            self.Requirements.client_tags = ctc.Tag.LINUX_XENIAL

    def on_enqueue(self):
        self.Requirements.ramdrive = ctm.RamDrive(
            ctm.RamDriveType.TMPFS,
            self.Requirements.disk_space,
            None
        )

    def on_execute(self):
        self.__bootstrap = image.Bootstrap(
            ubuntu_release=self.Parameters.ubuntu_release,
            basic_includes_only=self.Parameters.custom_image and not self.Parameters.install_common,
        )
        self.__image = image.Image(
            self.Parameters.ubuntu_release,
            base_packages=self.Parameters.base_packages.split() if self.Parameters.custom_image else [],
            custom_packages=self.Parameters.custom_packages.split() if self.Parameters.custom_image else [],
        )

        with self.memoize_stage.create_image(commit_on_entrance=False):
            self.create_image()

        with self.memoize_stage.test_image(commit_on_entrance=False):
            image.test_image_statically(self.rootfs)

        if self.Parameters.test_result_lxc:
            with self.memoize_stage.run_tests(commit_on_entrance=False):
                self.run_tests()

        with self.memoize_stage.check_children(commit_on_entrance=False):
            self.check_children_state()

    def on_release(self, parameters):
        super(SandboxLxdImage, self).on_release(parameters)

    def find_resources(self):
        result = {}

        for resource_id, path in self.Parameters.resources.items():
            if resource_id.startswith("rbtorrent:"):
                skynet_id = resource_id
            else:
                resource = sdk2.Resource[resource_id]
                if resource is None:
                    raise common.errors.ResourceNotFound(resource_id)
                skynet_id = resource.skynet_id

            result[skynet_id] = path

        return result

    def add_resources_to_image(self):
        for skynet_id, path in self._resources.items():
            logging.info("Downloading resource {}".format(skynet_id))
            common.share.skynet_get(skynet_id, os.path.join(self.rootfs, path.lstrip(os.sep)))
            logging.info("Resource {} successfully downloaded".format(skynet_id))

    @property
    def release_template(self):
        if not self.Context.container_id or self.owner != ctu.SERVICE_GROUP:
            return super(SandboxLxdImage, self).release_template
        return sdk2.ReleaseTemplate(
            ["sandbox-releases"],
            sdk2.Resource[self.Context.container_id].description,
            self.Parameters.description,
        )
