import logging

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.task import SandboxTask
import datetime
from sandbox.sandboxsdk.process import run_process, check_process_return_code
from os.path import join as join_path, realpath, dirname


class UseDate(sdk_parameters.SandboxBoolParameter):
    name = 'use_dates'
    description = 'Use date'
    default_value = False
    required = True


class RunBatch(sdk_parameters.SandboxBoolParameter):
    name = 'run_batch'
    description = 'Run batch'
    default_value = True
    required = True


class WithTokensOnly(sdk_parameters.SandboxBoolParameter):
    name = 'with_tokens_only'
    description = 'Only load registrations with push tokens'
    default_value = False
    required = True


class BrowserTokens(sdk_parameters.SandboxBoolParameter):
    name = 'browser_tokens'
    description = 'Load browser registrations'
    default_value = False
    required = True


class MrClusterParam(sdk_parameters.SandboxStringParameter):
    name = 'cluster'
    description = 'YT cluster'
    default_value = 'hahn'
    required = True


class InputPathParam(sdk_parameters.SandboxStringParameter):
    name = 'input_path'
    description = 'YT input table path'
    default_value = '//statbox/metrika-mobile-log'
    required = True


class ResultsPathParam(sdk_parameters.SandboxStringParameter):
    name = 'results_path'
    description = 'YT results table path'
    default_value = '//home/search-functionality/sup/users'
    required = True


class VaultTokenItemNameParam(sdk_parameters.SandboxStringParameter):
    name = 'vault_item'
    description = 'Vault item name for YT token'
    default_value = 'mrtoken'
    required = True


class ApiKeyParam(sdk_parameters.ListRepeater, sdk_parameters.SandboxStringParameter):
    name = 'api_keys'
    description = 'API keys'
    default_value = ['10318', '10321', '42984', '42989', '86151']
    required = True


class ProcessingDateParam(sdk_parameters.SandboxStringParameter):
    name = 'input_date'
    description = 'Date (Y-m-d)'

    @property
    def default_value(self):
        yesterday = datetime.datetime.now() - datetime.timedelta(days=1)
        return yesterday.strftime('%Y-%m-%d')

    required = True


class ProcessingDaysParam(sdk_parameters.SandboxIntegerParameter):
    name = 'input_days'
    description = 'Days to analyze'
    default_value = 2
    required = True


class SupUrlParam(sdk_parameters.SandboxStringParameter):
    name = 'url'
    description = 'Sup url'
    required = True
    default_value = 'https://sup-prod.n.yandex-team.ru/registrations/batch/mongo'


class LimitParam(sdk_parameters.SandboxIntegerParameter):
    name = 'limit'
    description = 'Limit'


class ChunkSizeParam(sdk_parameters.SandboxIntegerParameter):
    name = 'chunk_size'
    description = 'Chunk size'
    required = True
    default_value = 2000


class SupRegistrationLoadTask(SandboxTask):
    type = 'SUP_REGISTRATION_LOAD'
    input_parameters = [InputPathParam, ResultsPathParam,
                        VaultTokenItemNameParam, ApiKeyParam, ProcessingDateParam, ProcessingDaysParam,
                        MrClusterParam, UseDate, SupUrlParam, LimitParam, ChunkSizeParam, RunBatch,
                        WithTokensOnly, BrowserTokens]

    def on_execute(self):
        # token = self.get_vault_data(self.owner, 'SUP_TOKEN')
        # if not token:
        #     raise errors.SandboxTaskFailureError('Empty token param')

        yesterday = (datetime.datetime.now() - datetime.timedelta(days=1)).strftime('%Y-%m-%d')
        dto = self.ctx[ProcessingDateParam.name] if self.ctx[UseDate.name] else yesterday
        results_path = join_path(self.ctx[ResultsPathParam.name], dto)

        with environments.VirtualEnvironment() as venv:
            venv.pip('yql')
            script_path = join_path(dirname(realpath(__file__)), 'request.py')
            yt_token = self.get_vault_data(self.owner, 'robot_sup_yql_token')
            input_path = self.ctx[InputPathParam.name]
            api_keys = ' '.join(self.ctx[ApiKeyParam.name])
            cluster = self.ctx[MrClusterParam.name]

            dfrom = (datetime.datetime.strptime(dto, '%Y-%m-%d').date() -
                     datetime.timedelta(days=self.ctx[ProcessingDaysParam.name] - 1)).strftime('%Y-%m-%d')

            process = run_process([
                venv.executable,
                script_path,
                '--output={}'.format(results_path),
                '--token={}'.format(yt_token),
                '--cluster={}'.format(cluster),
                '--input={}'.format(input_path),
                '--apikeys {}'.format(api_keys),
                '--dfrom={}'.format(dfrom),
                '--dto={}'.format(dto),
                '--with_tokens_only' if self.ctx[WithTokensOnly.name] else '',
                '--browser_tokens' if self.ctx[BrowserTokens.name] else '',
            ], log_prefix='request_yql', shell=True)
            check_process_return_code(process)

        if not self.ctx[RunBatch.name]:
            return

        import requests
        request = requests.post(self.ctx.get(SupUrlParam.name),
                                headers={
                                    # 'Authorization': 'OAuth %s' % token,
                                    'Content-Type': 'application/json'
                                },
                                json={k: v for k, v in
                                      {
                                          'path': results_path,
                                          'limit': self.ctx.get(LimitParam.name),
                                          'chunk_size': self.ctx.get(ChunkSizeParam.name)
                                      }.iteritems()
                                      if v is not None}
                                )
        if request.status_code != 200:
            raise errors.SandboxTaskFailureError('SUP API Error (%d):\n%s' % (request.status_code, request.content))

        logging.info('Batch upload result: ' + request.text)


__Task__ = SupRegistrationLoadTask
