# -*- coding: utf-8 -*-

import datetime
import logging
import os
import requests
import tempfile

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task

YT_RUNNER = 'yt_runner.py'
YT_PROXY = 'hahn.yt.yandex.net'
CHUNK_SIZE = 3000000


class SupUrlParam(parameters.SandboxStringParameter):
    name = 'url'
    description = 'Sup url'
    default_value = 'https://push-beta.n.yandex-team.ru'


class SupUsersParam(parameters.SandboxStringParameter):
    name = 'supUsers'
    description = 'Path to USP users table'
    required = True
    default_value = '//home/search-functionality/sup/users/%s'


class CryptaDevInfoParam(parameters.SandboxStringParameter):
    name = 'cryptaDevInfo'
    description = 'Path to Crypta device info table'
    required = True
    default_value = '//home/crypta/production/state/graph/dicts/dev_info'


class DstParam(parameters.SandboxStringParameter):
    name = 'dst'
    description = 'Dst prefix'
    required = True
    default_value = '//home/search-functionality/sup/tags/apps'


class WriteToMongoParam(parameters.SandboxBoolParameter):
    name = 'write_to_mongo'
    description = 'Write to mongo'
    default_value = False


class SupUpdateAppTags(task.SandboxTask):
    type = 'SUP_UPDATE_APP_TAGS'
    description = 'Update application tags for SUP users'
    input_parameters = [SupUrlParam, SupUsersParam, CryptaDevInfoParam, DstParam, WriteToMongoParam]
    environment = (
        environments.PipEnvironment('yandex-yt', '0.7.34-0'),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet')
    )

    def on_execute(self):
        sup_users = self.ctx[SupUsersParam.name]
        # if sup_users has '%s' in the string assume that yesterday date should
        # be substituted instead.
        if sup_users.find('%s') != -1:
            if sup_users.find(',') != -1:
                raise errors.SandboxTaskFailureError('List of sources is not supported with %s')
            # sup_users = sup_users % (datetime.date.today() - datetime.timedelta(days=1)).isoformat()
            sup_users = sup_users % datetime.date.today().isoformat()

        # to keep token private put it into temporary file, otherwise token will appear
        # in logs and even in web UI for the task on YT
        with tempfile.NamedTemporaryFile(bufsize=0) as token_file:
            token_file.write(self.get_vault_data(self.owner, 'robot_sup_hahn_token'))
            # run map/reduce in different Python to deal with Python-hell between
            # Sandbox and YT execution environments
            process.run_process(
                cmd=[
                    '/skynet/python/bin/python',
                    os.path.join(os.path.dirname(__file__), YT_RUNNER),
                    '--server=' + YT_PROXY,
                    '--token=@' + token_file.name,
                    '--sup_users=' + sup_users,
                    '--dev_info=' + self.ctx[CryptaDevInfoParam.name],
                    '--output=' + self.ctx[DstParam.name]
                ],
                log_prefix=YT_RUNNER,
                wait=True,
                check=True
            )

        if self.ctx.get(SupUrlParam.name):
            result = requests.post(
                os.path.join(self.ctx.get(SupUrlParam.name),
                             'tags/batch' if not self.ctx.get(WriteToMongoParam.name) else 'tags/batch/mongo'),
                headers={
                    'Content-Type': 'application/json'
                },
                json={'path': self.ctx.get(DstParam.name), 'chunk_size': CHUNK_SIZE}
            )

            if result.status_code != 200:
                raise errors.SandboxTaskFailureError('SUP API Error (%d):\n%s' %
                                                     (result.status_code, result.content))

            logging.info('Batch upload result: ' + result.text)


__Task__ = SupUpdateAppTags
