# -*- coding: utf-8 -*-
import os
import datetime
from os.path import join as join_path, realpath, dirname
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.process import run_process, check_process_return_code


class SupUrlParam(parameters.SandboxStringParameter):
    name = 'url'
    description = 'Sup url'
    default_value = 'https://push-beta.n.yandex-team.ru'


class SrcParam(parameters.SandboxStringParameter):
    name = 'src'
    description = 'Src prefix'
    required = True
    default_value = 'home/search-functionality/news-push/sup/users_for_sup'


class DstParam(parameters.SandboxStringParameter):
    name = 'dst'
    description = 'Dst prefix'
    required = True
    default_value = 'home/search-functionality/sup/tags/geo'


class LimitParam(parameters.SandboxIntegerParameter):
    name = 'limit'
    description = 'Limit'


class ChunkSizeParam(parameters.SandboxIntegerParameter):
    name = 'chunk_size'
    description = 'Chunk size'
    required = True
    default_value = 3000000


class QueryParam(parameters.SandboxStringParameter):
    name = 'query'
    description = 'YQL Query'
    multiline = True
    required = True


class WriteToMongoParam(parameters.SandboxBoolParameter):
    name = 'write_to_mongo'
    description = 'Write to mongo'
    default_value = False


class SupUpdateGeoTags(task.SandboxTask):
    type = 'SUP_UPDATE_GEO_TAGS'
    input_parameters = [QueryParam, SrcParam, LimitParam, ChunkSizeParam, DstParam, SupUrlParam, WriteToMongoParam]

    def on_execute(self):
        token = self.get_vault_data(self.owner, 'robot_sup_yql_token')
        if not token:
            raise errors.SandboxTaskFailureError('Empty token param')
        src, dst, query = self.expand()

        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('yql', venv=venv, use_wheel=True).prepare()
            script_path = join_path(dirname(realpath(__file__)), 'yql_runner.py')
            process = run_process(
                [venv.executable, script_path, '--token={}'.format(token), '--query="{}"'.format(query)],
                log_prefix='yql', shell=True)
            check_process_return_code(process)

        if not self.ctx.get(SupUrlParam.name):
            return

        import requests
        request = requests.post(
            os.path.join(self.ctx.get(SupUrlParam.name),
                         'tags/batch' if not self.ctx.get(WriteToMongoParam.name) else 'tags/batch/mongo'),
            headers={
                'Authorization': 'OAuth %s' % token,
                'Content-Type': 'application/json'
            },
            json={k: v for k, v in
                  {
                      'path': dst,
                      'limit': self.ctx.get(LimitParam.name),
                      'chunk_size': self.ctx.get(ChunkSizeParam.name)
                  }.iteritems()
                  if v is not None}
        )

        if request.status_code != 200:
            raise errors.SandboxTaskFailureError('SUP API Error (%d):\n%s' % (request.status_code, request.content))

    def expand(self):
        query = self.ctx.get(QueryParam.name).strip()
        if not query:
            raise errors.SandboxTaskFailureError('Empty query param')
        today = datetime.datetime.now()
        yesterday = today - datetime.timedelta(days=1)
        query_placeholders = {
            '%TODAY%': today.strftime('%Y-%m-%d'),
            '%YEAR%': today.strftime('%Y'),
            '%MONTH%': today.strftime('%m'),
            '%DAY%': today.strftime('%d'),

            '%YESTERDAY%': yesterday.strftime('%Y-%m-%d'),
            '%YESTERDAY_YEAR%': yesterday.strftime('%Y'),
            '%YESTERDAY_MONTH%': yesterday.strftime('%m'),
            '%YESTERDAY_DAY%': yesterday.strftime('%d'),
        }
        dst = self.ctx.get(DstParam.name).strip()
        for key, value in query_placeholders.iteritems():
            dst = dst.replace(key, value)
        query_placeholders['%DST%'] = dst
        src = self.ctx.get(SrcParam.name).strip()
        for key, value in query_placeholders.iteritems():
            src = src.replace(key, value)
        query_placeholders['%SRC%'] = src
        for key, value in query_placeholders.iteritems():
            query = query.replace(key, value)
        return src, dst, query


__Task__ = SupUpdateGeoTags
