# coding: utf-8

import yaml

from sandbox import sdk2
from sandbox.common.types import client as ctc

from .support_stat import StorageMetrics
from ..utils import get_environments


QUERY = 'Queue: "CLOUDSUPPORT" and Status: Закрыт and Updated: >= now() - 3d'
QUERY_EXTENDED = 'Queue: "CLOUDSUPPORT" and Updated: >= now() - 3d'

CONFIGS = '''
yc_based:
  robot: robot-yc-support
  use_sla: True
  vault_owner: epsilond1
  st_vault_name: yc_startrek_token
  stat_vault_name: yc_statface_token
  path_statface: Adhoc/yc_support/support_stat
'''


class SupportStatException(Exception):
    """
    Base exception for SB-Task SupportStat
    """


class SupportStat(sdk2.Task):
    issues = {}

    class Requirements(sdk2.Requirements):
        disk_space = 1024
        environments = get_environments()
        client_tags = ~ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup('Component') as component:
            component.values['yc_based'] = component.Value(value='yc_based')

        base_query = sdk2.parameters.String("Query", default_value=QUERY, required=True)
        extra_query = sdk2.parameters.String("Extra Query", default_value=QUERY_EXTENDED, required=True)

    class Context(sdk2.Task.Context):
        config = {}
        url = 'https://st-api.yandex-team.ru/v2/myself'
        user_agent = 'sandbox/projects/SupportStat'
        per_page = 5000

    def _append_issue(self, date, issue):
        if not date:
            return
        date = date.split('T')[0]
        if date not in self.issues:
            self.issues[date] = []
        self.issues[date].append(issue)

    def on_execute(self):
        from startrek_client import Startrek

        self.set_info('Read config...')
        self.Context.config = yaml.load(CONFIGS).get(self.Parameters.component)

        if not self.Context.config:
            raise SupportStatException('{key} not found in config! process terminated.'.format(
                key=self.Parameters.component
            ))

        self.set_info('Trying load OAuth-token for startrek...')
        client = Startrek(useragent=self.Context.user_agent,
                          base_url=self.Context.url,
                          token=sdk2.Vault.data(self.Context.config.get('vault_owner'), self.Context.config.get('st_vault_name'))
                          )

        self.set_info('Trying find issues with query:\n {}'.format(self.Parameters.base_query))

        for issue in client.issues.find(self.Parameters.base_query, per_page=self.Context.per_page):
            self._append_issue(issue['resolvedAt'], issue)

        self.set_info('Create storage metrics...')
        store_for_base = StorageMetrics(
            issues=self.issues,
            token=sdk2.Vault.data(self.Context.config.get('vault_owner'), self.Context.config.get('stat_vault_name')),
            path_statface_report=self.Context.config.get('path_statface'),
            use_sla=self.Context.config.get('use_sla'),
        )

        self.set_info('Loaded {count} dates. Begin computing and uploading base statistic...'.format(
            count=len(self.issues.values())
        ))

        store_for_base.upload_base()
        del store_for_base
        self.issues.clear()

        self.set_info('Remove base statistic storage and preparing extended statistics.')
        self.set_info('Trying find issues with query:\n {}'.format(self.Parameters.extra_query))

        for issue in client.issues.find(self.Parameters.extra_query, per_page=self.Context.per_page):
            self._append_issue(issue['createdAt'], issue)

        self.set_info('Start compute and upload extended statistics')
        store_for_extended = StorageMetrics(
            issues=self.issues,
            token=sdk2.Vault.data(self.Context.config.get('vault_owner'), self.Context.config.get('stat_vault_name')),
            path_statface_report=self.Context.config.get('path_statface'),
        )

        store_for_extended.upload_extended()
        del store_for_extended

        self.set_info('Finished!')
