import re
import logging

import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.task as sdk_task
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import utils

from sandbox.projects.SuspiciousCommits import rules
from sandbox.projects.SuspiciousCommits.rules import diff_parse


logger = logging.getLogger(__name__)

# TODO: Move diff-resolver extraction logic to separate module
_DIFF_RESOLVER_MASK = "[diff-resolver:"


def _file_content_getter(path, revision):
    url = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/{}'.format(path)
    try:
        return sdk_svn.Svn.cat(url, revision=revision)
    except sdk_svn.SvnError:
        return None


class RevisionToCheck(sdk_parameters.SandboxIntegerParameter):
    name = 'revision_to_check'
    description = 'Revision to check'
    required = True


class SendEmail(sdk_parameters.SandboxBoolParameter):
    """
    Send email to commit author.

    Introduced mostly for debug reasons.
    """

    name = 'send_email'
    description = 'Send email to commit author'
    required = False
    default_value = True


class SuspiciousCommits(sdk_task.SandboxTask):
    """
    Task for checking commit on suspicious content.

    See DEVRULES-79 and related issues.
    """

    type = 'SUSPICIOUS_COMMITS'

    cores = 1

    input_parameters = [
        RevisionToCheck,
        SendEmail,
    ]

    environment = [
        sdk_environments.SvnEnvironment()
    ]

    execution_space = 10 * 1024

    def on_execute(self):
        revision = self.ctx.get(RevisionToCheck.name)
        arcadia_url = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia@{}'.format(revision)

        diff = sdk_svn.Svn.diff(arcadia_url, revision)
        diff_summary = sdk_svn.Svn.diff(arcadia_url, revision, summarize=True)
        author = self._get_commit_author(arcadia_url, revision)

        cc = 'suspicious-commits@yandex-team.ru'

        diff_context = diff_parse.parse_diff(diff, diff_summary)

        logger.info('Diff: %s', diff)
        for changed_file, changes in diff_context.file_2_diff.iteritems():
            logger.info('Changed file: %s\nContent: %s\n', changed_file, changes)

        debug_only_rules = True

        messages = []
        for rule in rules.RULES:
            match = rule.function(_file_content_getter, diff_context, revision)

            if match is None:
                logger.debug("Rule %s has no matches, skipped", rule.name())
                continue

            messages += [
                rule.message,
                "File: {}".format(match.file_name),
            ]

            if match.line:
                messages.append("Line: {}".format(match.line))

            if match.pattern:
                messages.append("Matching pattern: {}".format(match.pattern))

            messages.append("")

            if not rule.debug:
                debug_only_rules = False

        if messages:
            messages.append('Revision: https://a.yandex-team.ru/commit/{}'.format(revision))

            logger.info("\n\nMESSAGES:\n%s\n\n", "\n".join(messages))

            if utils.get_or_default(self.ctx, SendEmail):
                to = ['{}@yandex-team.ru'.format(author)] if not debug_only_rules else []
                logger.info('Sending email: to %s and %s', author, cc)
                channel.sandbox.send_email(
                    to,
                    [cc],
                    'Suspicious commit {} by {}'.format(revision, author),
                    '\n'.join(messages)
                )
            else:
                logging.info("Email sending is DISABLED")

    @staticmethod
    def _get_commit_author(arcadia_url, revision):
        revision_log = sdk_svn.Arcadia.log(arcadia_url, revision, limit=1)[0]
        logging.debug("Revision log: %s", revision_log)
        msg = revision_log["msg"]
        stripped_msg = msg.replace(" ", "")
        if _DIFF_RESOLVER_MASK in stripped_msg:
            rm = re.search(r"diff-resolver:([a-z0-9-]+)", stripped_msg)
            if rm:
                return rm.group(1)
        return sdk_svn.Arcadia.info(arcadia_url)['author']


__Task__ = SuspiciousCommits
