import textwrap

from .common import CPP_FILES, added_lines_text, Match, find_patterns_single_match, Rule


class DeprecatedLib(Rule):
    message = textwrap.dedent("""
        Please stop using deprecated and legacy code from library/cpp/deprecated, etc.
        """)

    def function(self, file_content_getter, diff_context, revision):
        patterns = [
            # for contrib/deprecated/, library/cpp/deprecated/
            '/deprecated/',
            # by agorodilov@ request
            '/legacy/',
        ]
        return find_patterns_single_match(
            file_2_diff=diff_context.file_2_diff,
            file_pattern=CPP_FILES,
            patterns=patterns,
        )


def _check_library(pattern, file_content_getter, file_2_diff, revision, is_library=True):
    for changed_file, diff in file_2_diff.iteritems():
        diff = added_lines_text(diff)
        if changed_file.endswith('ya.make') and pattern in diff.split():
            if is_library and 'LIBRARY(' in file_content_getter(changed_file, revision):
                return Match(
                    file_name=changed_file,
                )
    return None


class MapReduceLibInLibrary(Rule):
    message = textwrap.dedent("""
        You should not use mapreduce/lib in LIBRARY, you can use mapreduce/interface instead.

        REASONS:
        * mapreduce/lib is an implementation of mapreduce/interface for Hadoop, YaMR, RTMR and YT,
          it should be used only in PROGRAM because developer of the PROGRAM may want to
          use implementation only for particular system.
        * it takes eternity to build mapreduce/lib.
        """)

    def function(self, file_content_getter, diff_context, revision):
        return _check_library("mapreduce/lib", file_content_getter, diff_context.file_2_diff, revision)


class MapReduceClientInLibrary(Rule):
    message = textwrap.dedent("""
        You should not use mapreduce/yt/client in LIBRARY, you can use mapreduce/yt/interface instead.

        REASONS:
        * mapreduce/yt/client is an implementation of mapreduce/yt/interface, it should only be used in
          PROGRAM because developer of the PROGRAM may want to use alternative implementation of interface
          (well, if there was any).
        * it takes eternity to build mapreduce/yt/client
        """)

    def function(self, file_content_getter, diff_context, revision):
        return _check_library("mapreduce/yt/client", file_content_getter, diff_context.file_2_diff, revision)


class Search2116(Rule):
    message = textwrap.dedent("""
        Library kernel/region2country is deprecated (see SEARCH-2116),
        use kernel/geodb or library/geobase5 instead.
        """)

    def function(self, file_content_getter, diff_context, revision):
        return _check_library("kernel/region2country", file_content_getter, diff_context.file_2_diff, revision, is_library=False)
