import re
import textwrap

from .common import Rule, Match


class Py2Project(Rule):
    owners = [
        "alexeykruglov",
        "mvel",
    ]

    message = textwrap.dedent("""
        Use PY23 or Python 3 in new projects when possible.

        See https://clubs.at.yandex-team.ru/arcadia/20122 for details.
        """)

    def function(self, file_content_getter, diff_context, revision):
        for path, status in diff_context.path_2_status.items():
            if not (path.endswith('ya.make') and status == 'A'):
                continue

            diff = diff_context.file_2_diff[path]

            if not _is_py2(diff):
                continue

            peerdirs = _get_peerdirs(diff)
            if peerdirs:
                has_py2_peerdirs = False

                for peerdir in peerdirs:
                    content = file_content_getter(peerdir + '/ya.make', revision - 1)
                    if not content:
                        continue
                    if _is_py2(content):
                        has_py2_peerdirs = True

                if has_py2_peerdirs:
                    continue

            return Match(path)


def _is_py2(ya_make_content):
    if 'PY23_LIBRARY' in ya_make_content or 'PY3_LIBRARY' in ya_make_content or 'PROTO_LIBRARY' in ya_make_content:
        return False

    return (
        ('PY_LIBRARY' in ya_make_content) or
        (
            'LIBRARY' in ya_make_content
            and 'USE_PYTHON' in ya_make_content  # See DEVTOOLSSUPPORT-8557
        ) or
        (
            'LIBRARY' in ya_make_content
            and 'PY_SRCS' in ya_make_content  # See DEVTOOLSSUPPORT-8557
        ) or
        # ('SANDBOX_TASK' in ya_make_content) or
        ('PY_PROGRAM' in ya_make_content)
    )


def _get_peerdirs(ya_make_content):
    match = re.search(r'PEERDIR\((.+?)\)', ya_make_content, re.MULTILINE | re.DOTALL)
    if not match:
        return None
    lines = [x.lstrip('+').strip() for x in match.group(1).split('\n')]
    return [x for x in lines if x]
