import logging
import re


_BASE64_RE = re.compile(r'[A-Za-z0-9+/]{32,}')

ALL_FILES = re.compile(r'.*')
CPP_FILES = re.compile(r'.*\.(cpp|cxx|cc|CC|inc|c|C|h|H|hpp|HPP)$')
PY_FILES = re.compile(r'.*\.(py)$')


class Rule(object):

    def name(self):
        return self.__class__.__name__

    owners = [
        "mvel",
        "alexeykruglov",
        "pg",
    ]

    # Committer sees this message when the check is triggered
    message = None

    # Strict rules must not be broken
    # commit could be reverted when changes are not approved
    strict = False

    debug = False

    def function(self, file_content_getter, diff_context, revision):
        """
        Rule implementation. Needs to be redefined in subclass.
        Returns Match object or None
        """
        raise Exception('Not implemented')


class Match(object):
    file_name = None
    line = None
    pattern = None

    def __init__(
        self,
        file_name,
        line=None,
        pattern=None,
    ):
        self.file_name = file_name
        self.line = line
        self.pattern = pattern


def _added_lines(diff):
    for line in diff.split('\n'):
        if line.startswith('+') and not line.startswith('+++'):
            yield line[1:]


def added_lines_text(diff):
    return '\n'.join(_added_lines(diff))


def find_patterns_matches(
    file_2_diff,
    file_pattern,
    patterns,
    added_only=True,
    to_lower=False,
    skip_base64=False,
):
    """
    Find specified patterns in given context.

    Return first match as `Match` object or all of them
    if `find_all` is set.

    :param file_2_diff: per-file diff (map file_name -> diff text)
    :param patterns: patterns to check
    :param message: message for match
    :param added_only: check only added lines
    :param to_lower: lowercase diff before checking patterns
    :param find_all: return list of all matches instead of first one
    :param skip_base64: ignore lines matching base64 sequences that sometimes give false positives
    :return: When find_all is False, return first match or None.
        When find_all is True, return list of all matches (or empty list)
    """
    for changed_file, diff_text in file_2_diff.iteritems():
        if file_pattern is not None:
            if not file_pattern.search(changed_file):
                logging.debug('File %s was skipped as non-matching file pattern')
                continue

        for line in diff_text.split("\n"):
            original_line = line
            if to_lower:
                line = line.lower()

            if added_only and not line.startswith('+') or line.startswith('+++'):
                continue

            for pattern in patterns:
                if pattern not in line:
                    continue

                if skip_base64 and _BASE64_RE.search(line):
                    logging.debug("Line '%s' with match '%s' is too like base64 string, skip it", line, pattern)
                    continue

                logging.debug("DETECTED: '%s' in '%s'", pattern, line)
                match = Match(
                    file_name=changed_file,
                    line=original_line,
                    pattern=pattern,
                )

                yield match


def find_patterns_single_match(
    file_2_diff,
    file_pattern,
    patterns,
    added_only=True,
    to_lower=False,
    skip_base64=False,
):
    iter = find_patterns_matches(
        file_2_diff=file_2_diff,
        file_pattern=file_pattern,
        patterns=patterns,
        added_only=added_only,
        to_lower=to_lower,
        skip_base64=skip_base64,
    )
    return next(iter, None)
