# -*- coding: utf-8 -*-

import logging
import xmlrpclib

from sandbox.projects.MediaLib.shardmap import Shardmap
from sandbox.projects.MediaLib.iss_shardtracker import iss_shardtracker_api
from sandbox.projects.MediaLib.media_zk import zk_client
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

PRIEMKA_TASK = 'PRIEMKA_VIDEO_BASESEARCH_DATABASE'
SHARDTRACKER = ('shardtracker.search.yandex.net', 9100)

ZK_FLAGS = {
    'newdb': '/media-services/video/flags/newdb_video_sbase/on_production',
    # 'on_production': '/media-services/video/flags/newdb/on_production',
    'thumb': '/media-services/video/flags/production_vla_vidsth/on_production'
}


class VideoAccept:
    def __init__(self, parent_task, shardmap):
        self.shardmap = shardmap
        self.parent = parent_task
        self.shardmap_timestamp = self.shardmap.shardmap_timestamp
        self.shardmap_name = self.shardmap.shardmap_filename
        self.message = "Database: {0}\n".format(self.shardmap_name)
        self.message += "Timestamp: {0}\n\n".format(self.shardmap_timestamp)

    def _is_db_uploaded(self):
        """
        Проверяем что база выехала в newdb конфигурации
        """
        with zk_client() as zk:
            newdb_shardmap, _ = zk.get(ZK_FLAGS['newdb'])
            newdb_shardmap = newdb_shardmap.split()[0]
        logging.debug("Newdb shardmap ts: {}".format(newdb_shardmap))
        if self.shardmap_name != newdb_shardmap:
            raise SandboxTaskFailureError("Db hasn't been uploaded yet")

    def _is_thumbs_ready(self):
        """
        Проверяем что тумбы моложе шардмапа для переключения
        """
        with zk_client() as zk:
            thumbs_uploaded, _ = zk.get(ZK_FLAGS['thumb'])
            logging.debug("Current thumbs shardmap: {}".format(thumbs_uploaded))
        thumbs_shardmap = Shardmap(thumbs_uploaded.split()[2])
        logging.debug("Thumbs shardmap ts: {}".format(thumbs_shardmap.shardmap_timestamp))

        if thumbs_shardmap.shardmap_timestamp < self.shardmap.shardmap_timestamp:
            raise SandboxTaskFailureError("Thumbs aren't ready yet!")

    def _check_shardmap_sizes(self):
        """
        Проверка размера шарда
        """
        # Shard size limit (in Gb)
        lower_size_limit = 19
        upper_size_limit = 28

        shard_sizes = []

        with iss_shardtracker_api() as thrift_shardtracker_client:
            for shard_name in self.shardmap.shards:
                shard_info = thrift_shardtracker_client.getShardInfo(shard_name)
                shard_size = float(shard_info.shardInfo.full.size)/2**30
                shard_sizes.append(shard_size)

                if shard_size < lower_size_limit or shard_size > upper_size_limit:
                    raise SandboxTaskFailureError("Shard size is incorrect: {0:.2f}GB, shard {1}\n".format(shard_size, shard_name))

        shards_count = len(self.shardmap.shards)
        info = """
Shard sizes are OK.
Shards count: {shards_count}
Shard sizes(min/avg/max): {min:.2f}/{avg:.2f}/{max:.2f}GB
Total index size: {total_size:.2f}GB
""".format(
            min=min(shard_sizes),
            avg=sum(shard_sizes) / shards_count,
            max=max(shard_sizes),
            shards_count=shards_count,
            total_size=sum(shard_sizes)
        )

        self.parent.set_info(info, do_escape=False)
        logging.debug(info)
        self.message += info

    def __check_performance_single_task(self, task, proxy):
        tasks_run = 0

        local_info = "Performance check task: <a href='https://sandbox.yandex-team.ru/task/{0}/view'>Task {0}</a><br/>".format(task["id"])
        for task_name, task_id in task["ctx"]["analyze_tasks"].iteritems():
            subtask = proxy.get_task(task_id)
            median = subtask["ctx"]["new_stats"]["diff"]["shooting.rps_0.5"]
            tasks_run += 1

            local_info += "\tMedian diff {2:.2f}% \t{1:<18}\t<a href='https://sandbox.yandex-team.ru/task/{0}/view'>{0}</a><br/>".format(task_id, task_name.split(",")[-2:], median)
            if median < -10:
                raise SandboxTaskFailureError("Median diff exceeds -10%")
        self.message += local_info
        self.parent.set_info(local_info, do_escape=False)

        return tasks_run

    def _check_performance(self):
        r = xmlrpclib.ServerProxy('https://sandbox.yandex-team.ru/sandbox/xmlrpc')
        t = r.list_tasks({"status": "SUCCESS", "task_type": PRIEMKA_TASK, "show_childs": True})

        found_task = 0
        for task in t:
            wrong_task = 0

            if "new_basesearch_database_shards" not in task["ctx"]:
                continue

            for shard in task["ctx"]["new_basesearch_database_shards"].split(","):
                if shard.find(self.shardmap_timestamp) == -1:
                    wrong_task = 1
                    break

            if wrong_task:
                continue

            found_task += self.__check_performance_single_task(task, r)

        if not found_task:
            raise SandboxTaskFailureError("No valid {0} task found for {1}".format(PRIEMKA_TASK, self.shardmap_timestamp))

    def accept(self):
        logging.debug("Check newdb shardmap")
        self._is_db_uploaded()
        logging.debug("New database has been uploaded")

        logging.debug("Check thumbs on production")
        self._is_thumbs_ready()
        logging.debug("Thumbs are ok")

        logging.debug("Check db performance")
        self._check_performance()
        logging.debug("Performance is ok")

        logging.debug("Checking shard sizes")
        self._check_shardmap_sizes()
        logging.debug("Shard sizes are ok")

        channel.sandbox.send_email(
            'guba@yandex-team.ru',
            None,
            'Видеобаза {0} принята по производительности'.format(self.shardmap_name),
            self.message,
            'text/html',
            'utf-8'
        )

        logging.debug("Acceptance done")
