# -*- coding: utf-8 -*-

import logging

import os

import sandbox.common.types.task as ctt

from sandbox.projects import resource_types
from sandbox.projects.MediaLib.shardmap import SvnShardmapDir
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.task import SandboxTask

DASHBOARD_NAME = 'video_base'
RELEASE_TYPE = 'stable'
SHARDMAP_DEPLOY_RECIPE = 'deploy_video_database'
DEPLOY_TASK = 'DEPLOY_NANNY_DASHBOARD'
MONITOR_TASK = 'MEDIA_AFTER_DEPLOY_ACTIONS'

CC_USERS = [
    "search-releases",
    "videosearch-releases",
    "guba",
    "mnikulin"
]


class ArcadiaSvnShardDir(SandboxArcadiaUrlParameter):
    name = 'svn_shardmap_dir'
    description = 'Arcadia svn shards dir'
    default_value = 'arcadia:/robots/trunk/genconf/shardmaps/VIDEO/'
    required = True


class ShardmapResourceType(SandboxStringParameter):
    name = 'shardmap_resource_type'
    description = 'Shardmap resource type'
    choices = [(str(x), str(x)) for x in resource_types.__dict__]
    required = True
    default_value = 'VIDEO_BASE_SHARDMAP'


class SwitchVideoDatabase(SandboxTask):
    """Переключение большой базы Видео"""

    execution_space = 1024
    SE_TAGS = {'limit1': 1}
    type = 'SWITCH_VIDEO_DATABASE'
    input_parameters = ArcadiaSvnShardDir, ShardmapResourceType

    def initCtx(self):
        self.ctx['kill_timeout'] = 25 * 60 * 60

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.se_tag:
            sem_name = "{}/{}".format(self.type, self.se_tag)
            self.semaphores(ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=sem_name, capacity=self.SE_TAGS[self.se_tag])
                ]
            ))

    def on_execute(self):
        # We must import it after getting shard_tracker module
        from DBAccept import VideoAccept

        if 'deploy_task' not in self.ctx:
            svn_dir = SvnShardmapDir(self.ctx['svn_shardmap_dir'])
            latest_shardmap = svn_dir.get_latest_shardmap()

            self.set_info(latest_shardmap.shardmap_filename)
            logging.debug("Found latest shardmap: {}".format(latest_shardmap))

            resources = channel.sandbox.list_resources(
                resource_type=self.ctx['shardmap_resource_type'],
                status='READY',
                attribute_name='shardmap_file_name',
                attribute_value=latest_shardmap.shardmap_filename,
                limit=999
            )
            logging.debug("Found already published resources: {}".format(resources))

            if resources:
                logging.debug("Latest shardmap has been published.")
                return

            logging.debug("Run db acceptance")
            accepter = VideoAccept(self, shardmap=latest_shardmap)
            accepter.accept()
            logging.debug("Done db acceptance")
            self.ctx['shardmap_file_name'] = latest_shardmap.shardmap_filename

            create_shardmap_task_params = {
                'remote_file_protocol': 'svn',
                'remote_file_name': os.path.join(self.ctx['svn_shardmap_dir'], latest_shardmap.shardmap_filename),
                'created_resource_name': latest_shardmap.shardmap_filename,
                'resource_type': self.ctx['shardmap_resource_type'],
                'resource_attrs': "shardmap_file_name={}".format(latest_shardmap.shardmap_filename)

            }

            create_shardmap_subtask = self.create_subtask(task_type='CREATE_SHARDMAP_RESOURCE',
                                                          description=latest_shardmap.shardmap_filename,
                                                          input_parameters=create_shardmap_task_params)

            deploy_task_params = {
                'deployment_task_id': create_shardmap_subtask.id,
                'deployment_release_status': RELEASE_TYPE,
                'deployment_nanny_dashboard_name': DASHBOARD_NAME,
                'deployment_nanny_dashboard_recipe': SHARDMAP_DEPLOY_RECIPE
            }
            deploy_task = self.create_subtask(
                task_type=DEPLOY_TASK,
                description='Deploy video base shardmap',
                input_parameters=deploy_task_params
            )
            self.ctx['deploy_task'] = deploy_task.id

            self.wait_tasks([create_shardmap_subtask.id, deploy_task.id],
                            tuple(self.Status.Group.FINISH + self.Status.Group.BREAK),
                            wait_all=True)

        deploy_task = channel.sandbox.get_task(self.ctx['deploy_task'])

        if not deploy_task.is_finished():
            raise SandboxTaskFailureError("Bad deploy task status: {}".format(deploy_task.status))

        deploy_taskgroup_id = deploy_task.ctx.get('deploy_taskgroup', None)

        if deploy_taskgroup_id is None:
            raise SandboxTaskFailureError("Can't find deploy_taskgroup_id in {}".format(self.ctx['deploy_task']))

        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )

        nanny_ticket = nanny_client.create_ticket(
            queue_id='VIDEO',
            title='Переключение базы видео: {}'.format(self.ctx['shardmap_file_name']),
            description='''Некоторые детали описаны тут:
    https://beta.wiki.yandex-team.ru/jandekspoisk/sepe/docs/media/videobasedeploy/
Тикет:
    https://nanny.yandex-team.ru/ui/#/alemate/taskgroups/{}/table/'''.format(deploy_taskgroup_id),
            responsible="searchmon-notify",
            urgent=True,
            copy_to=CC_USERS
        )
        logging.debug("Created Nanny ticket: {}".format(nanny_ticket))
        self.ctx['nanny_ticket'] = nanny_ticket['value']['id']
        self.set_info(
            "Created Nanny ticket: <a href=https://nanny.yandex-team.ru/ui/#/t/{ticket}/>{ticket}</a>".format(
                ticket=self.ctx['nanny_ticket']), do_escape=False)

        watch_task_params = {
            'nanny_taskgroup_id': deploy_taskgroup_id,
            'nanny_ticket_to_close': self.ctx['nanny_ticket'],
            'after_deploy_zk_path': '/media-services/video/flags/vidsbase/on_production',
            'after_deploy_zk_data': '{} production_vidsbase_[a-z]*-1471184240'.format(self.ctx['shardmap_file_name'])
        }

        self.create_subtask(
            task_type=MONITOR_TASK,
            description="Wait VIDEO db switch",
            input_parameters=watch_task_params
        )


__Task__ = SwitchVideoDatabase
