import os
import re
import json
import shutil
import logging
import xml.etree.ElementTree as et

from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters


def find(path):
    for root, dirs, files in os.walk(path):
        for f in files:
            yield os.path.join(root, f)


def iter_const_files(real_paths, expected_paths):
    store = set(expected_paths)
    for p in real_paths:
        if p not in store:
            yield p


def iter_ivy_deps(ivy):
    try:
        for dep in ivy.find('dependencies').findall('dependency'):
            art = dep.get('name')
            ver = dep.get('rev')
            if art is not None and ver is not None:
                yield (art, ver)
    except AttributeError:
        pass


def iter_makelist_deps(makelist):
    m = makelist.replace('\n', ' ')
    try:
        for p in re.search(r'PEERDIR *\((?P<peerdir>.*?)\)', m).group('peerdir').strip().split():
            art = os.path.basename(os.path.dirname(p))
            ver = os.path.basename(p)
            yield (art, ver)
    except AttributeError:
        pass


def has_mismatch(makelist_path, ivy_path):
    ivy_deps = dict(list(iter_ivy_deps(et.parse(ivy_path).getroot())))
    makelist_deps = dict(list(iter_makelist_deps(open(makelist_path).read())))

    mismatch = False

    for art, ver in makelist_deps.iteritems():
        if art not in ivy_deps:
            logging.warning('Extra dep in %s: %s-%s', makelist_path, art, ver)
        elif ver != ivy_deps[art]:
            logging.error('Mismatch for %s:\n%s in %s\n%s in %s', art, ver, makelist_path, ivy_deps[art], ivy_path)
            mismatch = True

    for art, ver in ivy_deps.iteritems():
        if art not in makelist_deps:
            logging.warning('Missing dep in %s: %s-%s', makelist_path, art, ver)

    return mismatch


class SyncIceberg(task.SandboxTask):
    type = 'SYNC_ICEBERG_TASK'

    class Author(parameters.SandboxStringParameter):
        name = 'author'
        description = 'Vault owner'

    class Name(parameters.SandboxStringParameter):
        name = 'name'
        description = 'Vault name'

    class SvnUrl(parameters.SandboxStringParameter):
        name = 'svn_url'
        description = 'Svn url'

    class HgUrl(parameters.SandboxStringParameter):
        name = 'hg_url'
        description = 'Hg url'

    class Force(parameters.SandboxBoolParameter):
        name = 'force'
        description = 'Force'
        default_value = False

    input_parameters = [
        Author,
        Name,
        SvnUrl,
        HgUrl,
        Force,
    ]

    def on_execute(self):
        ya = svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', os.path.realpath('ya'))

        def hg(*args):
            with ssh.Key(self, self.ctx['author'], self.ctx['name']):
                process.run_process([ya, 'tool', 'hg'] + list(args), log_prefix='hg')

        svn_path = os.path.realpath('svn_source')
        svn.Arcadia.checkout(self.ctx['svn_url'], svn_path)

        try:
            id_ = open(os.path.join(svn_path, '.id')).read().strip()
        except IOError as e:
            raise errors.SandboxTaskFailureError(e.message)
        logging.info("current id is {}".format(id_))

        hg_path = os.path.realpath('hg_source')
        hg('clone', self.ctx['hg_url'], hg_path, '-r', id_)

        const_files = list(iter_const_files(
            [os.path.relpath(f, svn_path) for f in find(svn_path) if '.svn' not in f.split(os.path.sep)],
            [os.path.relpath(f, hg_path) for f in find(hg_path) if '.hg' not in f.split(os.path.sep)]
        ))
        logging.info("const files:\n{}".format(json.dumps(const_files, indent=4, sort_keys=True)))

        store = os.path.realpath('store')
        for f in const_files:
            try:
                os.makedirs(os.path.join(store, os.path.dirname(f)))
            except OSError:
                pass
            shutil.copy(os.path.join(svn_path, f), os.path.join(store, f))

        for p in os.listdir(svn_path):
            if '.svn' not in p:
                try:
                    os.remove(os.path.join(svn_path, p))
                except OSError:
                    pass

                try:
                    shutil.rmtree(os.path.join(svn_path, p))
                except Exception:
                    pass

        shutil.rmtree(hg_path)
        hg('clone', self.ctx['hg_url'], hg_path)

        for p in os.listdir(hg_path):
            if '.hg' not in p:
                try:
                    shutil.copy(os.path.join(hg_path, p), os.path.join(svn_path, p))
                except Exception:
                    pass

                try:
                    shutil.copytree(os.path.join(hg_path, p), os.path.join(svn_path, p))
                except Exception:
                    pass

        for f in const_files:
            if os.path.exists(os.path.join(svn_path, f)):
                raise errors.SandboxTaskFailureError('file {} exists in hg repository'.format(f))
            else:
                try:
                    os.makedirs(os.path.join(svn_path, os.path.dirname(f)))
                except OSError:
                    pass
                shutil.copy(os.path.join(store, f), os.path.join(svn_path, f))

        p = process.run_process([ya, 'tool', 'hg', 'id', '-i', hg_path], outs_to_pipe=True)
        latest_id = p.communicate()[0].strip()
        logging.info('latest id is {}'.format(latest_id))

        mismatch = False
        for f in const_files:
            if os.path.basename(f) == 'CMakeLists.txt':
                makelist_path = os.path.join(svn_path, f)
                ivy_path = os.path.join(svn_path, os.path.dirname(f), 'ivy.xml')
                if os.path.exists(ivy_path) and has_mismatch(makelist_path, ivy_path):
                    mismatch = True

        if mismatch:
            raise errors.SandboxTaskFailureError('Mismatches were found. See log for details.')

        if latest_id != id_:
            with open(os.path.join(svn_path, '.id'), 'w') as f:
                f.write(latest_id)

            st = svn.Arcadia.status(svn_path).strip()
            logging.info('status is:\n{}'.format(st))
            for x in st.split('\n'):
                if len(x.split()) != 2:
                    logging.warning(x)
                    continue
                state, f = x.split()
                if state == '?':
                    svn.Arcadia.add(f)
                elif state == '!':
                    svn.Arcadia.delete(f)

            st = svn.Arcadia.status(svn_path).strip()
            logging.info('new status is:\n{}'.format(st))

            m = 'AUTOCOMMIT({}): sync {} -> {}'.format(self.id, self.ctx['hg_url'], self.ctx['svn_url'])

            if self.ctx['force']:
                m += ' __FORCE_COMMIT__'

            with ssh.Key(self, self.ctx['author'], self.ctx['name']):
                svn.Arcadia.commit(svn_path, m, self.ctx['author'])


__Task__ = SyncIceberg
