import logging
import datetime
import os
import re
import collections
from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
from sandbox.common.types import client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox.projects.common.build import YaPackage
from sandbox.sdk2.helpers import subprocess


_YT_TOKEN_NAME = 'ROBOT_TAXI_SANDLOAD_YT_TOKEN'


class TaxiGraphBaseResource(sdk2.Resource):
    """
        Abstract base with necessary attributes for every taxi graph resources
    """

    version = sdk2.Attributes.String('version')
    yt_path = sdk2.Attributes.String('yt_path')

    def set_attributes(self, version, yt_path):
        self.version = version
        self.yt_path = yt_path


# OLD DEPRECATED resources begin
class TaxiGraphResourse(TaxiGraphBaseResource):
    """
        Resource that contains all graph files
    """
    pass


class TaxiGraphRoadGraphResourse(TaxiGraphBaseResource):
    """
        Resource that contains all files needed for road graph
    """
    pass


class TaxiGraphPersistentIndexResourse(TaxiGraphBaseResource):
    """
        Resource that contains all files needed for persistent index
    """
    pass


class TaxiGraphLeptideaResourse(TaxiGraphBaseResource):
    """
        Resource that contains all files needed for leptidea
    """
    pass
# OLD DEPRECATED resources end

# PROD resources begin
class TaxiGraphMainResourceProd(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all graph files
    """
    pass


class TaxiGraphRoadGraphResourceProd(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all files needed for road graph
    """
    pass


class TaxiGraphPersistentIndexResourceProd(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all files needed for persistent index
    """
    pass


class TaxiGraphLeptideaResourceProd(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all files needed for leptidea
    """
    pass
# PROD resources end


# TEST resources begin
class TaxiGraphMainResourceTest(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all graph files
    """
    pass


class TaxiGraphRoadGraphResourceTest(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all files needed for road graph
    """
    pass


class TaxiGraphPersistentIndexResourceTest(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all files needed for persistent index
    """
    pass


class TaxiGraphLeptideaResourceTest(TaxiGraphBaseResource):
    """
        Resource for test purposes that contains all files needed for leptidea
    """
    pass
# TEST resources end


FilesPackage = collections.namedtuple(
    'FilesPackage',
    [
        'resource_type',
        'files',
        'description',
    ]
)

PROD_STR = 'PROD'
TEST_STR = 'TEST'
OLD_DEPRECATED_STR = 'OLD_DEPRECATED'


MAIN_PACKAGE_DICT = {
    PROD_STR: TaxiGraphMainResourceProd,
    TEST_STR: TaxiGraphMainResourceTest,
    OLD_DEPRECATED_STR: TaxiGraphResourse,
}


ROAD_GRAPH_PACKAGE_DICT = {
    PROD_STR: TaxiGraphRoadGraphResourceProd,
    TEST_STR: TaxiGraphRoadGraphResourceTest,
    OLD_DEPRECATED_STR: TaxiGraphRoadGraphResourse,
}


PERSISTENT_INDEX_PACKAGE_DICT = {
    PROD_STR: TaxiGraphPersistentIndexResourceProd,
    TEST_STR: TaxiGraphPersistentIndexResourceTest,
    OLD_DEPRECATED_STR: TaxiGraphPersistentIndexResourse,
}


LEPTIDEA_PACKAGE_DICT = {
    PROD_STR: TaxiGraphLeptideaResourceProd,
    TEST_STR: TaxiGraphLeptideaResourceTest,
    OLD_DEPRECATED_STR: TaxiGraphLeptideaResourse,
}


class ResourcesSet(object):
    def __init__(self, set_str):
        self.main_package = self._init_main_package(set_str)
        self.additional_packages = self._init_additional_packages(set_str)

    @staticmethod
    def _init_main_package(set_str):
        return FilesPackage(
            resource_type = MAIN_PACKAGE_DICT[set_str],
            files=[],
            description='Taxi graph files',
        )

    @staticmethod
    def _init_additional_packages(set_str):
        additionla_packages_template = [
            FilesPackage(
                resource_type=ROAD_GRAPH_PACKAGE_DICT,
                files=[
                    'road_graph.fb',
                    'rtree.fb',
                ],
                description='road graph files',
            ),
            FilesPackage(
                resource_type=PERSISTENT_INDEX_PACKAGE_DICT,
                files=[
                    'edges_persistent_index.fb',
                ],
                description='persistent index',
            ),
            FilesPackage(
                resource_type=LEPTIDEA_PACKAGE_DICT,
                files=[
                    'l6a_data.fb.7',
                    'l6a_topology.fb.7',
                ],
                description='leptidea',
            )
        ]

        return [
            FilesPackage(
                resource_type=package_template.resource_type[set_str],
                files=package_template.files,
                description=package_template.description
            )
            for package_template in additionla_packages_template
        ]


class TaxiGraphAcceptanceTool(sdk2.Resource):
    version = sdk2.Attributes.String('version')
    platform = sdk2.Attributes.String('platform')


class TaxiGraphStatisticsTool(sdk2.Resource):
    version = sdk2.Attributes.String('version')
    platform = sdk2.Attributes.String('platform')


class GraphAcceptanceEnvironment(environments.TarballToolkitBase):
    resource_type = 'TAXI_GRAPH_ACCEPTANCE_TOOL'
    name = 'graph_acceptance_tool'
    sys_path_utils = ['graph_acceptance',]

    def prepare(self):
        env_dir = super(GraphAcceptanceEnvironment, self).prepare()
        logging.info('prepare acceptance env_dir = {}'.format(env_dir))
        self.update_os_path_env(env_dir)
        self.check_environment()
        return env_dir


class GraphStatisticsEnvironment(environments.TarballToolkitBase):
    resource_type = 'TAXI_GRAPH_STATISTICS_TOOL'
    name = 'graph_statistics_tool'
    sys_path_utils = ['graph_statistics',]

    def prepare(self):
        env_dir = super(GraphStatisticsEnvironment, self).prepare()
        logging.info('prepare statistics env_dir = {}'.format(env_dir))
        self.update_os_path_env(env_dir)
        self.check_environment()
        return env_dir


def _fail_if_subtask_failed(task):
    sub_tasks = task.find().limit(10)
    for sub_task in sub_tasks:
        if sub_task.status == ctt.Status.FAILURE:
            raise common.errors.TaskFailure('Sub task failed')

class CommonParameters(sdk2.Task.Parameters):
    yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
    with sdk2.parameters.String('Resource types set', required=True, default=OLD_DEPRECATED_STR) as resource_type:
        resource_type.values.PROD = PROD_STR
        resource_type.values.TEST = TEST_STR
        resource_type.values.OLD_DEPRECATED = OLD_DEPRECATED_STR



class TaxiGraphDoUploadTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            GraphAcceptanceEnvironment('0.1'),
            GraphStatisticsEnvironment('0.1'),
        )
        client_tags = ctc.Tag.LINUX_XENIAL
        ram = 100 * 1024 # graph loads to memory
        disk_space = 300 * 1024 # files occupy ~ 90GB, additional resources may occupy another 90GB

    class Parameters(CommonParameters):
        yt_path = sdk2.parameters.String('YT path to dir with files', required=True, default='//home/maps/graph/18.12.24-0')

    @staticmethod
    def _get_timestamp_from_path(path):
        _date_str = path.rsplit('/', 1)[1]
        _date = datetime.datetime.strptime(_date_str, '%y.%m.%d-%H')

        return str(int((_date - datetime.datetime(1970, 1, 1)).total_seconds()))

    def on_execute(self):
        yt = self._get_yt_client()
        yt_path = self.Parameters.yt_path
        logging.info('check path {} existance'.format(yt_path))
        if not yt.exists(yt_path):
            raise common.errors.TaskFailure('path {} does not exist'.format(yt_path))
        logging.info('path {} exists'.format(yt_path))

        timestamp = self._get_timestamp_from_path(yt_path)
        logging.info('timestamp from dir {}'.format(timestamp))

        resources_set = self._get_resources_set()

        try:
            created_resources_datas = []

            # create main resource
            main_resource_data = self._create_resource_data(
                resources_set.main_package,
                timestamp,
                yt_path
            )
            created_resources_datas.append(main_resource_data)
            main_resource_dir = str(main_resource_data.path)
            self._fill_main_resource(main_resource_dir, yt, yt_path)

            self._run_tests_and_fail_if_tests_fails(main_resource_dir)

            # create all additional resources
            for package in resources_set.additional_packages:
                resource_data = self._create_resource_data(
                    package,
                    timestamp,
                    yt_path
                )
                created_resources_datas.append(resource_data)
                resource_dir = str(resource_data.path)
                self._fill_additional_resource(
                    resource_dir,
                    package.files,
                    main_resource_dir
                )

            self._mark_ready(created_resources_datas)
            # save timestamp for graph-switch
            self.Context.files_version = timestamp
            return
        except Exception as e:
            logging.error('Problems with some resource: {}'.format(e))
            self._mark_broken(created_resources_datas)
            raise

    def _get_yt_client(self):
        yt_proxy = self.Parameters.yt_proxy
        logging.info('Use YT {}'.format(yt_proxy))
        import yt.wrapper as yt
        yt.config.set_proxy(yt_proxy)
        yt.config['token'] = sdk2.Vault.data(_YT_TOKEN_NAME)
        return yt

    def _get_resources_set(self):
        resource_set_str = self.Parameters.resource_type  # TODO wrong parameter
        return ResourcesSet(resource_set_str)

    @staticmethod
    def _fill_main_resource(resource_dir, yt, yt_path):
        logging.info('start upload files from YT {}'.format(yt_path))
        files = yt.list(yt_path)
        logging.info('all files: {}'.format(', '.join(files)))
        for file_name in files:
            resource_file_path = os.path.join(resource_dir, file_name)
            with open(resource_file_path, 'w') as f:
                yt_file_path = os.path.join(yt_path, file_name)
                logging.info('start upload {} to {}'.format(yt_file_path, resource_file_path))
                for chunk in yt.read_file(yt_file_path).chunk_iter():
                    f.write(chunk)
                logging.info('upload of {} done successfully'.format(yt_file_path))
        logging.info('finish upload')

    def _run_tests_and_fail_if_tests_fails(self, main_resource_dir):
        acceptance_result = self._acceptance(main_resource_dir)
        statistics_result = self._statistics(main_resource_dir)

        if not acceptance_result or not statistics_result:
            raise common.errors.TaskFailure('Graph files or library broken, see graph_acceptance.*.log and graph_statistics.*.log for details')

    def _create_resource_data(self, package, timestamp, yt_path):
        resource_type = package.resource_type
        resource = resource_type(
            self,
            package.description,
            self._resource_dir(resource_type, timestamp),
        )
        resource.set_attributes(timestamp, yt_path)
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(parents=True)
        return resource_data

    @staticmethod
    def _fill_additional_resource(resource_dir, files, main_resource_dir):
        for file_name in files:
            existed_file_path = os.path.join(main_resource_dir, file_name)
            sdk2.paths.copy_path(
                existed_file_path,
                resource_dir
            )
            logging.info('copy file {} from {} to {}'.format(
                file_name,
                main_resource_dir,
                resource_dir,
            ))

    @staticmethod
    def _mark_ready(resource_datas):
        for resource_data in resource_datas:
            resource_data.ready()

    @staticmethod
    def _mark_broken(resource_datas):
        for resource_data in resource_datas:
            resource_data.broken()

    @staticmethod
    def _resource_dir(resource_type, timestamp):
        return os.path.join(str(resource_type), timestamp)

    def _acceptance(self, path):
        with sdk2.helpers.ProcessLog(self, logger='graph_acceptance') as pl:
            check = subprocess.Popen(
                ['graph_acceptance', path],
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            check.wait()
            logging.info('graph_acceptance returncode = {}'.format(check.returncode))

            return check.returncode == 0

    def _statistics(self, path):
        with sdk2.helpers.ProcessLog(self, logger='graph_statistics') as pl:
            check = subprocess.Popen(
                ['graph_statistics', path],
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            check.wait()
            logging.info('graph_statistics returncode = {}'.format(check.returncode))

            return check.returncode == 0


class TaxiGraphUploadTask(sdk2.Task):
    class Parameters(TaxiGraphDoUploadTask.Parameters):
        """
            This task has the same parameters that TaxiGraphDoUploadTask has
        """
        pass

    def on_execute(self):
        with self.memoize_stage.build_environments:
            subtasks = [
                self._create_and_enqueue_package_task(
                    'taxi/graph/tools/graph_acceptance/pkg.json',
                    'TAXI_GRAPH_ACCEPTANCE_TOOL'
                ),
                self._create_and_enqueue_package_task(
                    'taxi/graph/tools/graph_statistics/pkg.json',
                    'TAXI_GRAPH_STATISTICS_TOOL'
                ),
            ]
            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, wait_all=True)
        with self.memoize_stage.do_upload:
            _fail_if_subtask_failed(self)
            self._set_versions_to_tools_resources()
            do_upload_task = TaxiGraphDoUploadTask(
                self,
                description=self.Parameters.description,
                notifications=self.Parameters.notifications,
            )
            do_upload_task.Parameters.yt_proxy = self.Parameters.yt_proxy
            do_upload_task.Parameters.yt_path = self.Parameters.yt_path
            do_upload_task.Parameters.resource_type = self.Parameters.resource_type
            do_upload_task.save()
            do_upload_task.enqueue()
            raise sdk2.WaitTask(do_upload_task, ctt.Status.Group.FINISH, wait_all=True)
        with self.memoize_stage.pack_switch:
            sub_tasks = self.find(TaxiGraphDoUploadTask).limit(1)
            for do_upload_task in sub_tasks:
                if do_upload_task.status == ctt.Status.FAILURE:
                    raise common.errors.TaskFailure('TAXI_GRAPH_DO_UPLOAD_TASK failed')
                files_version = do_upload_task.Context.files_version
                logging.info('files version {}'.format(files_version))
                pack_switch_task = self._create_and_enqueue_pack_switch_task(files_version)
                raise sdk2.WaitTask(pack_switch_task, ctt.Status.Group.FINISH, wait_all=True)
        with self.memoize_stage.finish:
            _fail_if_subtask_failed(self)

    def _create_and_enqueue_package_task(self, pkg_path, resource_type):
        task_type = sdk2.Task['YA_PACKAGE']
        kwargs = {
            YaPackage.PackagesParameter.name: pkg_path,
            YaPackage.ResourceTypeParameter.name: resource_type,
            YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
        }
        child = task_type(
            self,
            **kwargs
        )
        requirements = {
            'platform': 'xenial',
        }
        sdk2.Task.server.task[child.id].update({'requirements': requirements})

        child.enqueue()
        return  child

    def _set_versions_to_tools_resources(self):
        sub_tasks = self.find().limit(10)
        for sub_task in sub_tasks:
            task_resources = sdk2.Resource.find(
                task=sub_task,
            ).limit(10)
            for resource in task_resources:
                if resource.type in [
                        'TAXI_GRAPH_ACCEPTANCE_TOOL',
                        'TAXI_GRAPH_STATISTICS_TOOL'
                        ]:
                    resource.version = '0.1'

    def _create_and_enqueue_pack_switch_task(self, files_version):
        task_type = sdk2.Task['YA_PACKAGE']
        kwargs = {
            YaPackage.PackagesParameter.name: 'taxi/graph/packages/taxigraphswitch/pkg.json',
            YaPackage.CustomVersionParameter.name: files_version,
            YaPackage.PublishPackageParameter.name: True,
            YaPackage.PublishToParameter.name: 'yandex-taxi-common'
        }
        child = task_type(
            self,
            **kwargs
        )
        child.enqueue()
        return child

class TaxiGraphUploadLaunchTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
        )

    class Parameters(CommonParameters):
        yt_parent_path = sdk2.parameters.String('YT path to dir with files', required=True, default='//home/maps/graph')

    def on_execute(self):
        with self.memoize_stage.launch_upload:
            yt_current_path = self._get_yt_current_path()
            logging.info('find last graph dir "{}"'.format(yt_current_path))
            if not self._need_update(yt_current_path):
                logging.info('Do not need to download graph files')
                return
            logging.info(
                'Resource with files from {} does not exists, start download'
                .format(yt_current_path)
            )
            upload_task = self._create_and_enqueue_upload_task(yt_current_path)
            raise sdk2.WaitTask(upload_task, ctt.Status.Group.FINISH, wait_all=True)
        with self.memoize_stage.check_subtask:
            _fail_if_subtask_failed(self)
            self._send_notifications_about_data_update()

    def _send_notifications_about_data_update(self):
        subject = self._create_message_subject()
        message = self._create_message()
        for notification in self.Parameters.notifications:
            if notification.transport == common.types.notification.Transport.EMAIL:
                self._send_message_via_email(subject, message, notification.recipients)
                continue
            if notification.transport == common.types.notification.Transport.TELEGRAM:
                self._send_message_via_telegram(subject, message, notification.recipients)
                continue
            if notification.transport == common.types.notification.Transport.Q:
                self._send_message_via_q(subject, message, notification.recipients)
                continue
            logging.error('Unknown transport {}. Try send email to {}'.format(
                notification.transport,
                notification.recipients,
            ))
            self._send_message_via_email(subject, message, notification.recipients)

    def _send_message_via_email(self, subject, message, recipients):
        logging.info('Try send email to {} with subject "{}" and body "{}"'.format(
            recipients,
            subject,
            message,
        ))
        try:
            self.server.notification(
                subject=subject,
                body=message,
                recipients=recipients,
                transport=common.types.notification.Transport.EMAIL,
                urgent=True,
            )
            logging.info('Success send email')
        except common.rest.Client.HTTPError as e:
            logging.error('Failed to send email, exception {}'.format(e))

    def _send_message_via_telegram(self, subject, message, recipients):
        full_message = '\n'.join((subject, message))
        logging.info('Try send message "{}" via telegram to {}'.format(
            full_message,
            recipients,
        ))
        try:
            self.server.notification(
                body=full_message,
                recipients=recipients,
                transport=common.types.notification.Transport.TELEGRAM,
            )
            logging.info('Success send message via telegram')
        except common.rest.Client.HTTPError as e:
            logging.error('Failed to send message via telegram, exception {}'.format(e))

    def _send_message_via_q(self, subject, message, recipients):
        full_message = '\n'.join((subject, message))
        logging.info('Try send message "{}" via Q to {}'.format(
            full_message,
            recipients,
        ))
        try:
            self.server.notification(
                body=full_message,
                recipients=recipients,
                transport=common.types.notification.Transport.Q,
            )
            logging.info('Success send message via Q')
        except common.rest.Client.HTTPError as e:
            logging.error('Failed to send message via telegram, exception {}'.format(e))

    def _create_message_subject(self):
        return 'New graph files version available'

    def _create_message(self):
        message = [
            'new files available, graph-downloader will start download soon',
            'sandbox task ' + str(self.Context.__GSID),
        ]
        return '\n'.join(message)

    def _get_yt_current_path(self):
        import yt.wrapper as yt

        def check_yt_path(path):
            if not yt.exists(path):
                raise common.errors.TaskFailure('path {} does not exist'.format(path))

        def get_latest_dir(parent_path):
            latest_link = parent_path + '/latest&'  # without & list_attributes will return attributes of link target, not link itself
            target_path = yt.get_attribute(latest_link, 'target_path')
            logging.info('{} link points to {}'.format(latest_link, target_path))
            return target_path

        yt_proxy = self.Parameters.yt_proxy
        yt_parent_path = self.Parameters.yt_parent_path
        yt.config.set_proxy(yt_proxy)
        yt.config['token'] = sdk2.Vault.data(_YT_TOKEN_NAME)
        check_yt_path(yt_parent_path)

        yt_current_path = get_latest_dir(yt_parent_path)
        check_yt_path(yt_current_path)

        return yt_current_path

    def _need_update(self, path):
        return self._find_resources_with_yt_path(path).count == 0

    def _find_resources_with_yt_path(self, path):
        resource_type = self._get_main_resource_type()
        result = resource_type.find(
            state=ctr.State.READY,
            attrs=dict(yt_path=path)
        ).limit(1)
        logging.info('find {} resources contains files from {}'.format(
            result.count,
            path
        ))
        return result

    def _get_main_resource_type(self):
        resources_set_str = self.Parameters.resource_type
        resources_set = ResourcesSet(resources_set_str)
        return resources_set.main_package.resource_type

    def _create_and_enqueue_upload_task(self, yt_current_path):
        upload_task = TaxiGraphUploadTask(
            self,
            description=self.Parameters.description,
        )
        upload_task.Parameters.yt_proxy = self.Parameters.yt_proxy
        upload_task.Parameters.yt_path = yt_current_path
        upload_task.Parameters.resource_type = self.Parameters.resource_type
        upload_task.save()
        upload_task.enqueue()
        return upload_task
