import requests
import logging
import os.path
import sys
import json
import datetime
import time
from dateutil.relativedelta import relativedelta
from sandbox.common.types.misc import NotExists
from sandbox import sdk2
from multiprocessing import cpu_count
from multiprocessing.pool import ThreadPool

LOGS_LIMIT = 1024 * 1024 * 10
STATUS_OK = 200
STATUS_UNAUTHORIZED = 401
TIMEOUT = 60
EMIT_INTERVAL_SECONDS = 5
MAX_BUFFER_SIZE = 200


class AgentlessBase:
    class CustomHandler(logging.StreamHandler):

        def __init__(self, server_url, auth, build_id, parent):
            logging.StreamHandler.__init__(self)
            self.server_url = server_url
            self.build_id = build_id
            self.auth = auth
            self.limit = LOGS_LIMIT
            self.parent = parent
            self.buffer = []
            self.last_emit = datetime.datetime.now()

        def emit(self, record):
            if self.limit <= 0:
                self.parent.upload_logs = True
                return
            msg = self.format(record)
            msg = self.change_escaped_values(msg)
            if record.levelname == 'INFO' or record.levelname == 'DEBUG':
                msg_fmt = "##teamcity[message text='{0}']"
            elif record.levelname == 'WARNING':
                msg_fmt = "##teamcity[message text='{0}' status='WARNING']"
            elif record.levelname == 'ERROR':
                msg_fmt = "##teamcity[message text='{0}' status='ERROR']"
            else:
                msg_fmt = "##teamcity[message text='{0}' status='FAILURE']"
            msg_to_send = msg_fmt.format(msg)
            self.limit -= sys.getsizeof(msg_to_send)
            try:
                self.buffer.append(msg_to_send)
                if datetime.datetime.now() > self.last_emit + relativedelta(seconds=EMIT_INTERVAL_SECONDS) or \
                   len(self.buffer) > MAX_BUFFER_SIZE:
                    _ = requests.post(
                        '{0}/app/rest/builds/id:{1}/log'.format(self.server_url, self.build_id),
                        headers=self.auth,
                        data=''.join(self.buffer)
                    )
                    self.buffer = []
                    self.last_emit = datetime.datetime.now()
            except Exception as e:
                print(e)

        def change_escaped_values(self, msg):
            replacements = {"|": "||", "'": "|'", "\n": "|n", "\r": "|r", "[": "|[", "]": "|]"}
            escaped = []
            for ch in msg:
                if ch in replacements:
                    escaped.append(replacements[ch])
                else:
                    escaped.append(ch)
            return ''.join(escaped)

    class Decorators:
        @staticmethod
        def agentless_create_safe(logger):
            def agentless(func):
                def decorated(*args, **kwargs):
                    self = args[0]
                    self.ctx = self.Context.sandbox_task_launcher_context
                    if self.ctx is NotExists or "agentless_server_url" not in self.ctx:
                        func(*args, **kwargs)
                        return
                    self.logger = logger
                    self._set_auth()
                    logger.addHandler(
                        self.CustomHandler(self.ctx["agentless_server_url"], self.auth,
                                           self.ctx["agentless_build_id"], self)
                    )
                    try:
                        with self.memoize_stage.agentless_step(commit_on_entrance=False,
                                                               commit_on_wait=False):
                            self._open_logs_block()
                            func(*args, **kwargs)
                    except Exception as e:
                        self._close_logs_block()
                        raise e
                    except (sdk2.WaitTask, sdk2.Wait) as e:
                        self._close_logs_block()
                        raise e
                    time.sleep(EMIT_INTERVAL_SECONDS)
                    self.logger.info('')
                    self._close_logs_block()
                    self.finish()
                    logger.removeHandler(logger.handlers[-1])

                return decorated

            return agentless

        @staticmethod
        def agentless_create(func):
            def decorated(*args, **kwargs):
                self = args[0]
                self.ctx = self.Context.sandbox_task_launcher_context
                if self.ctx is NotExists or "agentless_server_url" not in self.ctx:
                    func(*args, **kwargs)
                    return
                self._set_auth()
                func(*args, **kwargs)
                self._send_finish_request()
            return decorated

        @staticmethod
        def add_logger(logger):
            def agentless(func):
                def decorated(*args, **kwargs):
                    self = args[0]
                    self.logger = logger
                    if self.ctx is NotExists or "agentless_server_url" not in self.ctx:
                        func(*args, **kwargs)
                        return
                    logger.addHandler(
                        self.CustomHandler(self.ctx["agentless_server_url"], self.auth, self.ctx["agentless_build_id"], self)
                    )
                    func(*args, **kwargs)
                    logger.removeHandler(logger.handlers[-1])
                return decorated
            return agentless

    def _set_auth(self):
        token = sdk2.Vault.data(self.owner, 'teamcity_token')
        self.auth = {"Authorization": "OAuth {}".format(token)}

    def _open_logs_block(self):
        self._send_log_request(
            "##teamcity[blockOpened name='Sandbox logs' description='Logs gotten from sandbox']"
        )

    def _close_logs_block(self):
        self._send_log_request("##teamcity[blockClosed name='Sandbox logs']")

    def _change_build_id(self, new_build_id):
        self.logger.warning("Changing build number")
        self._send_log_request("##teamcity[buildNumber '{0}']".format(new_build_id))

    def _notify_on_failure(self):
        self._send_log_request("##teamcity[buildStatus status='FAILURE' text='Sandbox task failed']")

    def _notify_on_success(self):
        self._send_log_request('Success, finishing!')

    def _add_build_comment(self, comment):
        _ = requests.put(
            "{0}/app/rest/builds/id:{1}/comment".format(self.ctx["agentless_server_url"],
                                                        self.ctx["agentless_build_id"]),
            headers=self.auth,
            data=comment
        )

    def _send_log_request(self, data):
        try:
            _ = requests.post(
                '{0}/app/rest/builds/id:{1}/log'.format(self.ctx["agentless_server_url"],
                                                        self.ctx["agentless_build_id"]),
                headers=self.auth,
                data=data
            )
        except Exception as e:
            print(e)

    def _upload_resource(self, resource):
        if not hasattr(self, "ctx") or self.ctx is NotExists or "agentless_server_url" not in self.ctx:
            self.logger.warning("Context is not defined")
            return
        self.logger.info("Uploading resource {}".format(resource.id))
        data = sdk2.ResourceData(resource)
        path = str(data.path)

        if os.path.isfile(path):
            self._upload_file(path)
        elif os.path.isdir(path):
            self._upload_dir(path)

    def _set_upload_resource(self, resource):
        if resource.state != "READY":
            self.logger.info("Resource {} is not ready".format(resource.id))
            return
        if self.Context.resources is NotExists:
            self.Context.resources = [resource.id]
        else:
            self.Context.resources += [resource.id]
        self.Context.save()

    def _upload_file(self, path, additional_path=''):
        self.logger.info("Uploading file {}".format(path))
        f = open(path, 'rb')
        filename = path.split('/')[-1]
        try:
            if "use_s3" in self.ctx:
                response = requests.post(
                    '{0}/artifactUploadS3.html'.format(self.ctx["agentless_server_url"]),
                    headers=self.auth,
                    data=json.dumps({'buildId': self.ctx["agentless_build_id"],
                                     'filePaths': [str(additional_path) + "/" +
                                                   filename if len(additional_path) > 0 else filename],
                                     'sizes': [os.path.getsize(path)]})
                )
                if response.status_code != STATUS_OK:
                    self.logger.warning("Getting links using json data failed with status {}, trying another option"
                                        .format(response.status_code))
                    response = requests.post(
                        '{0}/artifactUploadS3.html'.format(self.ctx["agentless_server_url"]),
                        headers=self.auth,
                        data={'buildId': self.ctx["agentless_build_id"],
                              'filePaths': [str(additional_path) + "/" +
                                            filename if len(additional_path) > 0 else filename],
                              'sizes': [os.path.getsize(path)]}
                    )
                if response.status_code != STATUS_OK:
                    self.logger.warning("Uploading file status {}".format(response.status_code))
                    self.logger.warning("Uploading file content {}".format(response.content))
                    raise Exception
                link = json.loads(response.content)[0]
                requests.put(
                    link,
                    data=f
                )
                return
            response = requests.post(
                '{0}/artifactUpload.html'.format(self.ctx["agentless_server_url"]),
                headers=self.auth,
                files={filename: f},
                data={'buildId': self.ctx["agentless_build_id"], 'path': str(additional_path)}
            )
            if response.status_code != STATUS_OK:
                self.logger.warning("Uploading file status {}".format(response.status_code))
                raise Exception
        except Exception as e:
            self.logger.info(e)
            raise Exception

    def _upload_dir(self, path, additional_path=''):
        n = len(path)
        files = {}
        for dirname, dirnames, filenames in os.walk(path):
            for file in filenames:
                if "use_s3" in self.ctx:
                    path = dirname[n:] + "/" + file
                    if path[0] == "/":
                        path = path[1:]
                    if len(additional_path) > 0:
                        path = additional_path + "/" + path
                    files[dirname + "/" + file] = path
                    continue
                files[dirname[n:] + "/" + file] = open(dirname + "/" + file, "rb")
        if "use_s3" in self.ctx:
            file_paths = []
            sizes = []
            for real_path, new_path in sorted(files.items()):
                file_paths.append(new_path)
                sizes.append(os.path.getsize(real_path))
            response = requests.post(
                '{0}/artifactUploadS3.html'.format(self.ctx["agentless_server_url"]),
                headers=self.auth,
                data=json.dumps({'buildId': self.ctx["agentless_build_id"],
                                 'filePaths': file_paths,
                                 'sizes': sizes})
            )
            if response.status_code != STATUS_OK:
                self.logger.warning("Getting links using json data failed with status {}, trying another option"
                                    .format(response.status_code))
                response = requests.post(
                    '{0}/artifactUploadS3.html'.format(self.ctx["agentless_server_url"]),
                    headers=self.auth,
                    data={'buildId': self.ctx["agentless_build_id"],
                          'filePaths': file_paths,
                          'sizes': sizes}
                )
            if response.status_code != STATUS_OK:
                self.logger.warning("Uploading directory status {}".format(response.status_code))
                self.logger.warning("Uploading directory content {}".format(response.content))
                raise Exception
            links = json.loads(response.content)
            pool = ThreadPool(cpu_count())
            pool.map(AgentlessBase._upload_s3_via_presigned_link, zip(links, sorted(files.items())))
            return

        response = requests.post(
            '{0}/artifactUpload.html'.format(self.ctx["agentless_server_url"]),
            headers=self.auth,
            files=files,
            data={'buildId': self.ctx["agentless_build_id"], 'path': str(additional_path)}
        )
        if response.status_code != STATUS_OK:
            self.logger.warning("Uploading directory status {}".format(response.status_code))
            self.logger.warning("Uploading directory content {}".format(response.content))
            raise Exception

    @staticmethod
    def _upload_s3_via_presigned_link(item):
        link, (path, _) = item
        _ = requests.put(
            link,
            data=open(path, "rb")
        )

    def __check_build_running(self):
        hdrs = {'Accept': 'application/json'}
        hdrs.update(self.auth)
        response = requests.get(
            '{0}/app/rest/builds/id:{1}'.format(self.ctx["agentless_server_url"], self.ctx["agentless_build_id"]),

            headers=hdrs
        )
        if response.status_code == STATUS_UNAUTHORIZED:
            self.logger.info("Credentials error")
            return False
        if response.status_code == STATUS_OK:
            data = response.json()
            if "state" in data and data["state"] == "finished":
                return False
        return True

    def _send_finish_request(self):
        running = self.__check_build_running()
        if not running:
            return
        if hasattr(self, "upload_logs"):
            logs = list(sdk2.Resource.find(task=self, type="TASK_LOGS", state="NOT_READY").limit(0))[0]
            self._upload_resource(logs)
        time.sleep(EMIT_INTERVAL_SECONDS)
        self.logger.info('')
        response = requests.put(
            '{0}/app/rest/builds/id:{1}/finish'.format(self.ctx["agentless_server_url"],
                                                       self.ctx["agentless_build_id"]),
            headers=self.auth
        )
        self.logger.info(response.status_code)
        if response.status_code != STATUS_OK:
            self.logger.info(response.status_code)
            self.logger.info(response.content)
            raise Exception

    def finish(self):
        resources = self.Context.resources
        try:
            running = self.__check_build_running()
            if not running:
                return

            if self.Context.resources is not NotExists:
                resources = self.Context.resources
                for resource_id in self.Context.resources:
                    resource = sdk2.Resource[resource_id]
                    self._upload_resource(resource)
                    resources = resources[1:]
            self._send_finish_request()
        except Exception as e:
            self.logger.info(e)
            self.Context.resources = resources
            self.Context.save()
            raise sdk2.WaitTime(TIMEOUT)
