# -*- coding: utf-8 -*-

import os
import logging
import datetime
import shutil
import glob
import subprocess
import traceback
import json
import threading
import time
import io
import select
import imp
import sys
import tempfile
import stat

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg
import sandbox.projects.common.constants as consts
import sandbox.projects.common.arcadia.sdk as sdk

from sandbox import common
from sandbox import sandboxsdk
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskUnknownError, SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder, remove_path
from sandbox.sandboxsdk.svn import Arcadia, ArcadiaTestData

# do not delete - needed for environments dynamic creation
from sandbox.projects.common.environments import *  # noqa
from sandbox.sandboxsdk.environments import *  # noqa

# 'devtools.fleur.util.path' copy/paste since there is no such
# good reuse technology like import hook


class ChangeDir(object):

    def __init__(self, *path):
        self.CurrentDir = os.getcwd()
        if path:
            os.chdir(os.path.join(*path))

    def __enter__(self):
        return self

    def __exit__(self, exceptionType, exceptionValue, traceback):
        os.chdir(self.CurrentDir)


class StreamFlusher(io.IOBase):

    def __init__(self, prefix, outputStream, lock, onDataCallback):
        self.Prefix = prefix
        self.Pipe = os.pipe()
        self.OutputStream = outputStream
        self.OutputLock = lock
        self.Thread = threading.Thread(target=self.ReadStream)
        self.OnData = onDataCallback
        self.Thread.start()

    def ReadStream(self):
        self.Running = True

        def read():
            buffer = b''
            for fh in select.select([self.Pipe[0]], [], [], 0)[0]:
                buffer += os.read(fh, 1024)
                while b'\n' in buffer:
                    data, buffer = buffer.split(b'\n', 1)
                    self.Flush(data.encode())

        while True:
            read()
            if not self.Running:
                read()
                break
            time.sleep(0.01)

    def Flush(self, data):
        try:
            self.OutputLock.acquire()
            self.OnData(data)
            self.OutputStream.write("{level}{message}\n".format(level=self.Prefix, message=data))
        finally:
            self.OutputLock.release()

    def fileno(self):
        return self.Pipe[1]

    def close(self):
        if self.Running:
            self.Running = False
            self.Thread.join()
            os.close(self.Pipe[0])
            os.close(self.Pipe[1])


def ListAbsPaths(folder):
    # regular files and .hidden ones
    return glob.glob(os.path.join(folder, "*")) + glob.glob(os.path.join(folder, ".*"))


def GetPatternsForDir(folder, patterns):
    candidates = []
    for pattern in patterns:
        candidates += glob.glob(os.path.join(folder, pattern))
    return candidates


def RemoveNonMatchingFiles(folder, filenames, patterns):
    # some links were links to files - have to delete them from the list, after RemoveLinks
    filenames = filter(os.path.exists, filenames)

    candidates = GetPatternsForDir(folder, patterns)
    for filename in filenames:
        path = os.path.join(folder, filename)
        if path not in candidates:
            os.remove(os.path.join(folder, path))


def RemoveLinks(folder):
    links = filter(os.path.islink, ListAbsPaths(folder))
    for link in links:
        os.unlink(os.path.join(folder, link))


def DirMatchesPattern(folder, patterns):
    return folder in GetPatternsForDir(os.path.dirname(folder), patterns)


class CommandExecutionException(Exception):
    pass


class TeamcityRunner(ArcadiaTask):

    type = 'TEAMCITY_RUNNER'

    execution_space = 100 * 1024  # 100GB

    class BuildProfile(parameters.SandboxSelectParameter):
        name = 'build_profile'
        description = 'Build profile'
        choices = [
            ('release', 'release'),
            ('debug', 'debug'),
        ]
        default_value = 'release'

    class Product(parameters.SandboxSelectParameter):
        name = 'product'
        description = 'Product'
        choices = [
            ('ORANGE', 'orange')
        ]
        default_value = 'orange'

    class CleanBuild(parameters.SandboxBoolParameter):
        name = 'clean_build'
        description = 'Run clean build (remove previous build files)'
        default_value = True

    class Debug(parameters.SandboxBoolParameter):
        name = 'debug'
        description = 'Run task in experimental mode'

    input_parameters = ArcadiaTask.input_parameters + [
        BuildProfile,
        Product,
        CleanBuild,
        Debug,
    ]

    DUPLOAD_CONF = {
        'search': {
            'fqdn': "search.dupload.dist.yandex.ru",
            'method': "scpb",
            'incoming': "/repo/search/mini-dinstall/incoming/",
            'dinstall_runs': 1,
        },
        'search-test': {
            "fqdn": "search-test.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/search-test/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'search-precise': {
            "fqdn": "search-precise.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/search-precise/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'yandex-precise': {
            "fqdn": "yandex-precise.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/yandex-precise/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'search-lucid': {
            "fqdn": "search-lucid.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/search-lucid/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'yandex-lucid': {
            "fqdn": "yandex-lucid.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/yandex-lucid/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'search-trusty': {
            "fqdn": "search-trusty.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/search-trusty/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'yandex-trusty': {
            "fqdn": "yandex-trusty.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/yandex-trusty/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'yabs-precise': {
            "fqdn": "yabs-precise.dupload.bsdist.yandex.net",
            "method": "scpb",
            "incoming": "/repo/yabs-precise/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'yabs-trusty': {
            "fqdn": "yabs-precise.dupload.bsdist.yandex.net",
            "method": "scpb",
            "incoming": "/repo/yabs-trusty/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        },
        'animals-precise': {
            "fqdn": "animals-precise.dupload.dist.yandex.ru",
            "method": "scpb",
            "incoming": "/repo/animals-precise/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        }
    }

    FakeSvnRoot = None
    ArcadiaDir = None

    def initCtx(self):
        self.ctx['build_bundle'] = False
        self.ctx['notify_via'] = 'email'
        self.ctx['if_failed'] = ''
        self.ctx['if_finished'] = ''
        self.ctx['fail_on_any_error'] = True

        # Temporary workaround for very slow deploy to Kiwi
        self.ctx['kill_timeout'] = 8 * 60 * 60

    def do_execute(self):
        self.Execute()

    def LogInfo(self, message):
        logging.info(message)
        if self.TeamCityLog:
            self.TeamCityLog.Info(message)

    def LogError(self, message):
        logging.error(message)
        if self.TeamCityLog:
            self.TeamCityLog.Error(message)

    def Execute(self):
        '''
        Perform all the steps according to 'sandbox.task.actions' value in the exact order.
        '''
        try:
            # Initialize may raise exception,
            # therefore declaration and initialization are separated.
            # Thus Finalize method can safely check the values of variables
            self.DeclareVariables()
            self.Initialize()
            failOnError = self.GetContextValue("fail.on.error", False)

            startTime = datetime.datetime.now()
            if self.GetContextValue("svn.root"):
                self.LogInfo("Checking out ...")
                self.Checkout()
                endTime = datetime.datetime.now()
                self.LogInfo("Check out completed in %s." % (endTime - startTime))

            for action in self.GetContextValue('actions.parsed'):
                actionName = action.capitalize()
                self.LogInfo("Executing handler '%s' action ..." % action)
                actionLogName = "Action %s" % actionName
                self.TeamCityLog.BlockOpen(actionLogName)

                startTime = datetime.datetime.now()
                if actionName.startswith("Cleanup"):
                    actionArg = 'all'
                    if "." in actionName:
                        actionName, actionArg = actionName.split(".", 1)
                    success = getattr(self, actionName)(actionArg)
                elif hasattr(self, actionName):
                    success = getattr(self, actionName)()
                else:
                    success = self.ExecuteCustomAction(action)
                endTime = datetime.datetime.now()
                delta = endTime - startTime
                self.LogInfo("Action '%s' completed in %s." % (action, delta))
                self.CustomMetricsDict["Action%sTime" % action.capitalize()] = delta.seconds * 1000  # milliseconds for charts
                self.TeamCityLog.BlockClose(actionLogName)
                if not success:
                    errorMessage = "%s failed" % actionName
                    self.TeamCityLog.BuildFailed(errorMessage)
                    if failOnError:
                        raise SandboxTaskFailureError(errorMessage)
                    else:
                        break
        finally:
            try:
                self.DumpCustomMetrics()
                self.Finalize()
            except:
                logging.error("Error while finalizing: %s" % traceback.format_exc())

    def ExecuteCustomAction(self, action):
        """
        Executes custom action.

        Add ACTION_NAME to "sandbox.task.actions" parameter to add a action to perform.
        Action should be presented with the format "sandbox.task.ACTION_NAME.command".
        You can also specify parameter "sandbox.task.ACTION_NAME.onerror"
        to prevent task failure if action command exited with non-zero exit code.
        """

        with ChangeDir(self.GetLogPath()):
            try:
                cmd = self.GetContextValue(action + ".command")
                try:
                    self.ExecuteAndProcessLogs(cmd, prefix=action)
                except CommandExecutionException:
                    if self.GetContextValue(action + ".onerror", "fail") == "fail":
                        raise
            except Exception:
                self.LogError(traceback.format_exc())
                return False

        return True

    def Checkout(self):
        # This call checkouts sources or updates existing one to the required
        # version. Sources location is returned and  stored in the task for the use.
        teamCityProgressMessage = "Svn checkout"
        self.TeamCityLog.BlockOpen(teamCityProgressMessage)
        start = datetime.datetime.now()

        # Some tools require a path to the arcadia based sources not from 'arcadia' folder,
        # but from the <branch> folder where 'arcadia' sits.
        # Since SandBox has a set of arcadia cached folders those names differ from 'arcadia',
        # e.g. 'arcadia_cache3', we need a fake folder that mimics like a root having 'arcadia' inside.
        # 'arcadia' is a symlink to the current arcadia cache folder.
        #
        # Tools now can reference such folder with {ROOT} pattern and this guaranties the under
        # the {ROOT} there is a folder (symlink) pointing to 'arcadia'
        if self.should_checkout():
            self.ArcadiaDir = sdk.do_clone(self.ctx.get(consts.ARCADIA_URL_KEY), self)
        else:
            self.ArcadiaDir = self.get_arcadia_src_dir()
        patch = self.ctx.get('arcadia_patch')
        if patch:
            Arcadia.apply_patch(self.ArcadiaDir, patch, self.abs_path())
        self.FakeSvnRoot = os.path.join(self.TestLogsDir, "SvnRoot")
        os.mkdir(self.FakeSvnRoot)
        logging.info("Creating arcadia symlink in '%s' pointing to '%s'" % (self.FakeSvnRoot, self.ArcadiaDir))
        os.symlink(self.ArcadiaDir, os.path.join(self.FakeSvnRoot, "arcadia"))

        testsDataPaths = self.GetContextValue("tests_data.paths")
        if testsDataPaths:
            testsDataPaths = testsDataPaths.split(",")
            for dataPath in testsDataPaths:
                dataPath = dataPath.strip()
                logging.info("Checking out tests data '%s'", dataPath)
                arcadiaTestDataSvnPath = os.path.normpath(Arcadia.append(self.GetContextValue('svn.root'), os.path.join("..", dataPath)))
                self.GetArcadiaTestData(arcadiaTestDataSvnPath)

            # test_data_location returns value is a tuple, with tests data root as first index
            arcadiaTestsDataLocation = ArcadiaTestData.test_data_location(arcadiaTestDataSvnPath)[0]
            testsDataLink = os.path.join(self.FakeSvnRoot, "arcadia_tests_data")
            logging.info("Creating symlink %s -> %s", arcadiaTestsDataLocation, testsDataLink)
            os.symlink(arcadiaTestsDataLocation, testsDataLink)

        delta = datetime.datetime.now() - start
        self.CustomMetricsDict["ActionCheckoutTime"] = delta.seconds * 1000  # milliseconds for charts
        self.TeamCityLog.BlockClose(teamCityProgressMessage)

    def GetArcadiaTestData(self, arcadiaTestDataSvnPath):
        maxAttempts = 5
        for attempt in range(maxAttempts):
            try:
                return ArcadiaTestData.get_arcadia_test_data(self, arcadiaTestDataSvnPath)
            except common.errors.TemporaryError:
                if attempt == (maxAttempts - 1):
                    raise

            time.sleep(5)

    def Finalize(self):
        """
        Ensure that all the code was executed
        """
        if self.VaultDir:
            remove_path(self.VaultDir)

        self.LogInfo("Sandbox task has completed")

    def postprocess(self):
        logging.info('Running custom postprocess')
        self.TeamCityLog = None
        if self.ArcadiaDir:
            shutil.rmtree(self.FakeSvnRoot)
        ArcadiaTask.postprocess(self)

    def DeclareVariables(self):
        '''
        Perform declaration of variables, because we cannot override __init__ method
        '''
        self.CustomMetricsDict = {}
        self.TestLogsDir = None
        self.VaultDir = None
        self.TeamCityLog = None

    def Initialize(self):
        '''
        Perform context parameters validation and initialized required fields.
        '''

        # Context parameters required for any action
        self.VerifyContext('product.name')
        self.VerifyContext('actions')

        actions = self.GetContextValue('actions').split(',')
        self.SetContextValue('actions.parsed', actions)

        # Verify presented custom actions
        for action in actions:
            actionName = action.capitalize()
            if not hasattr(self, actionName) and not action.startswith("cleanup"):
                self.VerifyCustomAction(action)

        # Check parameters specific for each action (some can be checked several times
        # to be more explicit and avoid problems when verify relevant to another action
        # is removed)
        if 'build' in actions:
            self.VerifyContext('build.type')
            self.VerifyContext('build.path')
            self.VerifyContext('svn.root')
            self.VerifyContext('svn.revision')

        if 'test' in actions:
            self.VerifyContext('test.app')
            self.VerifyContext('test.path')

        if 'package' in actions:
            self.VerifyContext('package.app')
            self.VerifyContext('arts.patterns')
            self.VerifyContext('build.platform')

        if 'deploy' in actions:
            self.VerifyContext('deploy.app')

        # Override some parameters (protocol, user and revision) to change authorization
        # info from teamcity to sandbox one

        if self.GetContextValue('svn.root'):
            svnurl = Arcadia.replace(self.GetContextValue('svn.root'), revision=self.GetContextValue('svn.revision'))

            self.SetContextValue('svn.root', svnurl)
            self.ctx[consts.ARCADIA_URL_KEY] = svnurl

            revision, tag, branch = self.arcadia_info()
            self.SetContextValue('svn.tag',      tag)
            self.SetContextValue('svn.branch',   branch)

        self.TestLogsDir = self.GetLogPath()
        self.InitTeamCityLogger()

        self.DumpEnvVars()
        # Environment variables required
        # define place where tests should find berkanavt folder
        os.environ['BERKANAVT_DIR'] = '/place/sandbox-data/REALTIME_TEST_data/berkanavt'
        # special parameters for correct svn work in case of key changes
        os.environ['SVN_SSH'] = os.environ.get("SVN_SSH", "ssh -q") + " -o StrictHostKeyChecking=no"

        for constructor in filter(None, self.GetContextValue('environments', "").split(";")):
            self.LogInfo("Preparation of environment: {0}".format(constructor))
            environment = eval(constructor.strip())
            environment.prepare()

    def InitTeamCityLogger(self):
        teamCityLogWriteSvnUrl = Arcadia.trunk_url('devtools/fleur/ytest/integration/teamcity.py')
        teamCityLogWriterModuleExportPath = os.path.join(self.TestLogsDir, "teamcity.py")
        Arcadia.export(teamCityLogWriteSvnUrl, teamCityLogWriterModuleExportPath)
        teamCityLogWriterModule = imp.load_source("teamcity", teamCityLogWriterModuleExportPath)
        self.TeamCityLog = teamCityLogWriterModule.TeamCityLogWriter(os.path.join(self.TestLogsDir, "TeamCity.log"), flowId="Sandbox Task")

    def Cleanup(self, cleanBuildDir):
        self.LogInfo("Removing non-version controlled files in '%s'..." % self.GetSourceDir())

        with ChangeDir(self.GetSourceDir()):
            cmd = 'svn stat --no-ignore| awk \'$1 == "?" || $1 == "I" { print $2 }\' | xargs rm -vrf'
            process = subprocess.Popen(
                cmd, shell=True, universal_newlines=True,
                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            output = process.communicate()
            retcode = process.poll()

            if retcode == 0:
                self.LogInfo("Done.")
            else:
                self.LogInfo("Command '%s' completed with error code %d." % (cmd, retcode))
                self.LogInfo("stdour: \n" + output[0])
                self.LogInfo("stderr: \n" + output[1])
                self.LogError(output[1])

            if cleanBuildDir == 'all':
                self.DeleteBuildDir()
            elif cleanBuildDir == 'bin':
                self.CleanBinaries()
            else:
                self.LogError("Unsupported argument for Cleanup action: %s" % cleanBuildDir)
                return False

        return True

    def RemovePath(self, filePath):
        if os.path.islink(filePath):
            self.RemovePath(os.path.join(os.path.dirname(filePath), os.readlink(filePath)))  # symlinks can be relative, make them absolute
        self.LogInfo("Removing path %s" % filePath)
        # After target remove link is broken so we have to use .lexists()
        if os.path.lexists(filePath):
            os.remove(filePath)

    def CleanBinaries(self):
        buildDir = self.GetBuildDir()
        if not os.path.exists(buildDir):
            return

        binDir = self.GetBinDir()
        libDir = self.GetLibDir()
        for binaryDir in [binDir, libDir]:
            if not os.path.exists(binaryDir):
                continue
            self.LogInfo("CleanBinaries in %s" % binaryDir)
            for filePath in os.listdir(binaryDir):
                filePath = os.path.join(binaryDir, filePath)
                if os.path.isfile(filePath) or os.path.islink(filePath):
                    self.RemovePath(filePath)
                else:
                    self.LogError("Unsupported path type in binary dir %s (neither file nor link)" % filePath)
        self.CleanProtos()
        self.LogInfo("CleanBinaries done")

    def CleanProtos(self):
        buildDir = self.GetBuildDir(self.GetContextValue('build.type'))
        for root, dirs, files in os.walk(buildDir):
            for pb2 in [f for f in files if f.endswith("_pb2.py")]:
                self.RemovePath(os.path.join(root, pb2))

    def GetBinDir(self):
        return self.GetBuildDir(self.GetContextValue('build.type'), "bin")

    def GetLibDir(self):
        return self.GetBuildDir(self.GetContextValue('build.type'), "lib")

    def DeleteBuildDir(self):
        buildDir = self.GetBuildRootDir()
        if os.path.exists(buildDir) or os.path.islink(buildDir):
            self.LogInfo("Removing build dir '%s'..." % buildDir)
            if os.path.islink(buildDir):
                self.LogInfo("Build dir is link - remove link")
                os.unlink(buildDir)
            else:
                self.LogInfo("Build dir is not link - real delete")
                shutil.rmtree(buildDir)
            self.LogInfo("Done.")
        else:
            self.LogInfo("Building dir '%s' does not exist" % buildDir)

    def Build(self):
        try:
            buildArgs = self.GetContextValue('build.system.params', "")
            self.BuildWithYa(buildArgs)
        except Exception:
            self.LogError(traceback.format_exc())
            return False
        return True

    def GetYaPath(self):
        return self.GetSourceDir("ya")

    def GetLogPath(self, *path):
        return os.path.join(self.log_path(), *path)

    def FillPlaceholders(self, value):
        if type(value) not in [str, unicode]:
            return value

        patterns = {
            "BUILD": self.GetBuildDir,
            "SRC": self.GetSourceDir,

            # fake folder with arcadia symlink
            "ROOT": lambda: self.FakeSvnRoot,
            "LOGS": lambda: self.TestLogsDir,
            "BRANCH": lambda: self.GetBranchShortName(),
            "SYS_EXECUTABLE": lambda: sys.executable,
            "SANDBOX_TASK_ID": lambda: str(self.id),
            "VAULT": self.GetVaultDir,
            "WORK": self.GetWorkDir,

            # In the furture the proper function can be implemented
            "SYSINFO(ncpu)": lambda: self.GetSystemInfo('ncpu'),
        }

        for key in patterns:
            pattern = "{%s}" % key
            if pattern in value:
                # Lazy data acquisition
                # If pattern does not contain key - method won't be called
                replacement = patterns[key]()
                value = value.replace(pattern, replacement)

        return value

    def RunPackager(self):
        packageApp = self.GetContextValue('package.app')
        packageParams = self.GetContextValue('package.app.params', "")
        self.LogInfo("  package app      = '%s'" % packageApp)
        self.LogInfo("  package params   = '%s'" % packageParams)
        packageKeyUserName = self.GetContextValue("package.key.user.name")
        packageKeyVaultOwnerName = self.GetContextValue("package.key.vault.owner")
        if packageKeyUserName and packageKeyVaultOwnerName:
            logging.info("Using %s user keys owned in vault by %s" % (packageKeyUserName, packageKeyVaultOwnerName))
            for distName in self.DUPLOAD_CONF:
                self.DUPLOAD_CONF[distName]["login"] = packageKeyUserName
            with sandbox.projects.common.gnupg.GpgKey(self, packageKeyVaultOwnerName, "%s-gpg-private" % packageKeyUserName, "%s-gpg-public" % packageKeyUserName):
                with sandbox.projects.common.debpkg.DebRelease(self.DUPLOAD_CONF):
                    with sandboxsdk.ssh.Key(self, packageKeyVaultOwnerName, "%s-ssh" % packageKeyUserName):
                        environment = os.environ.copy()
                        environment['DEBEMAIL'] = '%s@yandex-team.ru' % packageKeyUserName
                        self.ExecuteAndProcessLogs(" ".join([packageApp, packageParams]), prefix='package', env=environment)
        else:
            self.ExecuteAndProcessLogs(" ".join([packageApp, packageParams]), prefix='package')

        packagesMetaFile = "packages.json"
        if os.path.exists(packagesMetaFile):
            with open(packagesMetaFile) as metaFile:
                packages = json.loads(metaFile.read())
        else:
            packages = {}
            productName = self.GetContextValue('product.name')
            packageResultFileMask = [mask.strip() for mask in self.GetContextValue('package.result.mask', "*").split(",")]
            self.LogInfo("  package results  = '%s'" % packageResultFileMask)
            for mask in packageResultFileMask:
                for packagePath in glob.glob(mask):
                    packages[productName] = {"package": os.path.abspath(packagePath)}
        return packages

    def GetPackageResoureType(self):
        return self.GetContextValue('resource.type', default="TEAMCITY_RESOURCE")

    def PublishResources(self, packages, packageDir, packageTemp):
        self.LogInfo("Going to publish %s" % packages)
        for productName in packages:
            product = packages[productName]
            packagePath = product["package"]

            resourceType = product.get("resource", None)
            resourceType = resourceType or product.get("sandbox_type", None)
            resourceType = resourceType or self.GetPackageResoureType()
            resourceAttributes = product.get("attributes", {})

            self.LogInfo("  resource type     = '%s'" % resourceType)
            self.LogInfo("  package object    = '%s'" % packagePath)

            with ChangeDir(packageDir):
                # Move prepared package to data folder root to have exactly name we want
                # (without any additional folder levels)
                shutil.move(os.path.join(packageTemp, packagePath), packageDir)
                productVersion = product.get("version", None)

                self.SavePackage(productName, os.path.basename(packagePath), resourceType, productVersion, resourceAttributes)
                self.LogInfo("Package '%s' successfully created." % packagePath)

                if "content" in product:
                    packagePath = product["content"]
                    self.LogInfo("  package content    = '%s'" % packagePath)

                    # Move prepared package to data folder root to have exactly name we want
                    # (without any additional folder levels)
                    shutil.move(os.path.join(packageTemp, packagePath), packageDir)

                    self.RemoveLinks(os.path.basename(packagePath))
                    self.SavePackage("%s_content" % productName, os.path.basename(packagePath), "TEAMCITY_RESOURCE_PLAIN")
                    self.LogInfo("Package '%s' successfully created." % packagePath)

    def RemoveLinks(self, path):
        logging.info("Looking for symlinks in %s" % path)
        for root, _, fileNames in os.walk(path):
            for fileName in fileNames:
                filePath = os.path.join(root, fileName)
                logging.debug("Checking %s" % filePath)
                if os.path.islink(filePath):
                    logging.info("Removing link by %s" % filePath)
                    os.unlink(filePath)

    def Package(self):
        '''
        Produce a tar file ready for release, using package.py
        '''
        packageDir = self.abs_path()
        packageTemp = self.abs_path('package')

        self.LogInfo("Creating package...")
        self.LogInfo("  package dir      = '%s'" % packageDir)
        self.LogInfo("  package temp dir = '%s'" % packageTemp)

        try:
            failureMessage = "Failed to make package"
            make_folder(packageTemp)

            with ChangeDir(packageTemp):
                packages = self.RunPackager()
                self.LogInfo("  created packages  = '%s'" % packages)

                failureMessage = "Failed to publish package"
                self.PublishResources(packages, packageDir, packageTemp)

            remove_path(packageTemp)
        except:
            message = "%s '%s'" % (failureMessage, traceback.format_exc())
            self.ReportErrors("package", message)
            self.TeamCityLog.BuildStatistic("Infrastructure.PackageFailure", 1)
            self.TeamCityLog.BuildFailed(message)
            return False

        return True

    def ReportErrors(self, logFileName, message):
        self.LogInfo(message)
        self.LogError(traceback.format_exc())
        errLog = self.log_path("%s.err.txt" % logFileName)
        if os.path.exists(errLog):
            with open(errLog) as processLog:
                errLogMessage = processLog.read().decode('utf8', errors='replace')
                self.LogError("%s: %s" % (errLog, errLogMessage))
                message = os.linesep.join([message, errLogMessage])
        self.set_info(message)

    def Deploy(self):
        """
        Deploy action
        """

        self.LogInfo("Deploying...")

        deployApp = self.GetContextValue('deploy.app')
        params = self.GetContextValue('deploy.app.params', '')
        logFileName = "deploy"

        self.LogInfo("deploy app = '%s'" % deployApp)
        self.LogInfo("params = '%s'" % params)

        try:
            self.ExecuteAndProcessLogs("%s %s" % (deployApp, params), prefix=logFileName)
            self.LogInfo("Deployment completed.")
        except Exception, e:
            message = "Failed to run deploy: %s" % e
            self.ReportErrors(logFileName, message)
            self.TeamCityLog.BuildStatistic("Infrastructure.DeploySystemFailure", 1)
            self.TeamCityLog.BuildFailed(message)
            return False

        return True

    def UpdateMetricsFromFile(self, report):
        if os.path.exists(report):
            with open(report) as f:
                metrics = json.load(f)
            # Supporting old versions of files:
            # <if> is to be removed soon
            isOldVersionReport = not isinstance(metrics, dict)
            if isOldVersionReport:
                metricsPerSuite = metrics
                aggregatedMetrics = {}
            else:
                metricsPerSuite = metrics.get("MetricsPerSuite", [])
                aggregatedMetrics = metrics.get("AggregatedMetrics", {})
            for suite in metricsPerSuite:
                for counter in suite['Counters']:
                    self.CustomMetricsDict["%s.%s" % (suite['Name'], counter['Name'])] = counter['Value']
            for name, value in aggregatedMetrics.items():
                self.CustomMetricsDict[name] = value

        else:
            self.LogInfo('Cannot get performance report "%s"' % report)

    def Test(self):
        '''
        Run Tests Suite
        '''

        testPath = self.GetContextValue('test.path')
        testApp = self.GetContextValue('test.app')
        logFileName = "test"
        productName = self.GetContextValue('product.name')
        params = self.GetContextValue('test.app.params', '')
        testKeyUserName = self.GetContextValue("test.key.user.name")
        testKeyVaultOwnerName = self.GetContextValue("test.key.vault.owner", "TEAMCITY")
        # fail the test stage it there are test failures or other test errors
        testFail = self.GetContextValue("test.fail", default=False)

        self.LogInfo("Testing '%s' product..." % productName)
        self.LogInfo("  test path  = '%s'" % testPath)
        self.LogInfo("  test app   = '%s'" % testApp)
        self.LogInfo("  params     = '%s'" % params)
        testResult = False

        try:
            with ChangeDir(testPath):
                testCommand = "%s %s" % (testApp, params)

                if testKeyUserName:
                    self.LogInfo("  ssh user   = '%s'" % testKeyUserName)
                    with sandboxsdk.ssh.Key(self, testKeyVaultOwnerName, "%s-ssh" % testKeyUserName):
                        res = self._subprocess(testCommand, wait=True, check=False, log_prefix=logFileName)
                else:
                    res = self._subprocess(testCommand, wait=True, check=False, log_prefix=logFileName)

                if res.returncode == 0 or not testFail:
                    testResult = True

                if res.returncode not in [0, 1]:  # 0 - success, 1 - some tests are failed
                    raise Exception("Command '%s' failed with code %s" % (testCommand, res.returncode))

                self.LogInfo("product '%s' tests completed." % productName)
                perfomanceReport = os.path.join(self.TestLogsDir, '%s.report.performance' % os.path.splitext(os.path.basename(self.GetContextValue('test.app')))[0])
                self.UpdateMetricsFromFile(perfomanceReport)
        except Exception, e:
            message = "Failed to test product '%s'. %s" % (productName, e)
            self.ReportErrors(logFileName, message)
            self.TeamCityLog.BuildStatistic("Infrastructure.TestSystemFailure", 1)
            self.TeamCityLog.BuildFailed(message)
        finally:
            self.CleanTestLogs()
            self.DumpTestDataList()
            if not testResult and testFail:
                raise SandboxTaskFailureError("Some tests have failed and test fail flag was checked. For details see test logs")

        return testResult

    def GetBranchName(self):
        return self.GetContextValue('svn.branch') or self.GetContextValue('svn.tag') or 'trunk'

    def GetBranchShortName(self):
        branch = self.GetBranchName()
        # branch is returned with some path prefix, e.g. 'branch'
        # we need only the branch name with is the name of the last folder in the branch path
        return branch.rsplit("/", 1)[-1]

    def SavePackage(self, productName, packagePath, resourceType, resourceVersion=None, resourceAttributes=None):
        '''
        Generate resource description and save it.
        Nomenclature is as follows: Package.Platform.Profile.Branch.Revision,
        e.g. "orange.FreeBSD7.Debug.Orange133.346274"
        '''
        buildType = self.GetContextValue('build.type')
        revision = self.GetContextValue('svn.revision')
        branch = self.GetBranchName()
        platform = self.GetContextValue('build.platform')

        self.LogInfo("The current dir '%s'..." % os.getcwd())
        self.LogInfo("Storing package '%s'..." % packagePath)
        self.LogInfo("  build type  = '%s'" % buildType)
        self.LogInfo("  resource    = '%s'" % resourceType)
        self.LogInfo("  revision    = '%s'" % revision)
        self.LogInfo("  branch      = '%s'" % branch)
        self.LogInfo("  platform    = '%s'" % platform)

        nameComponents = [productName, platform, buildType, branch, revision, ]

        descr = '.'.join(nameComponents)
        attributes = resourceAttributes or {}
        attributes.update({
            "resource_name": productName,
            "svn_path": self.GetContextValue('svn.root'),
            "svn_revision": revision,
            "build_type": buildType,
            "branch": branch,
            "platform": platform,
        })

        if resourceVersion:
            attributes["resource_version"] = resourceVersion

        # TTL value is specified in days
        attributes['ttl'] = int(self.GetContextValue('resource.ttl', default=30))

        self._create_resource(descr, packagePath, resourceType, complete=1, attrs=attributes)
        self.LogInfo("Package '%s' successfully stored." % packagePath)

    def BuildWithYa(self, command):
        ya = [
            self.GetYaPath(),
            command,
        ]
        self.ExecuteAndProcessLogs(ya, prefix='ymake')

    def CleanTestLogs(self):
        '''
        Copy all log files and outputs to task log folder. It will be available in TeamCity's page 'Sandbox Logs'
        '''
        logging.debug('Cleaning Test Logs!')
        logPatterns = [pattern.strip() for pattern in self.GetContextValue('arts.patterns').split(',')]
        logPatterns = filter(None, logPatterns)  # remove empty patterns

        if os.path.exists(self.TestLogsDir):
            self.CleanDir(self.TestLogsDir, logPatterns)
        else:
            self.LogError("Test logs directory %s does not exist. Cannot save test logs." % self.TestLogsDir)

    def CleanDir(self, root, patterns):
        if self.FakeSvnRoot and os.path.samefile(self.FakeSvnRoot, root):
            return

        root = os.path.abspath(root)
        RemoveLinks(root)
        if DirMatchesPattern(root, patterns):
            return

        dirs = filter(os.path.isdir, ListAbsPaths(root))
        files = filter(os.path.isfile, ListAbsPaths(root))
        RemoveNonMatchingFiles(root, files, patterns)
        for dirname in dirs:
            self.CleanDir(os.path.join(root, dirname), patterns)
        if not os.listdir(root):
            if os.path.islink(root):
                os.unlink(root)
            else:
                os.rmdir(root)

    def ExecuteAndProcessLogs(self, command, prefix, env=None):
        if not env:
            env = os.environ.copy()
        if type(command) == list:
            command = " ".join(command)

        logPath = self.GetLogPath()
        lock = threading.Lock()
        info = "[info]"
        error = "[error]"
        lenInfo = len(info)
        lenError = len(error)
        jointFile = os.path.join(logPath, "%s.log" % prefix)

        executeBlockName = "Executing: %s" % prefix
        self.TeamCityLog.BlockOpen(executeBlockName)

        def onCommandOutputInfo(message):
            self.LogInfo(message)

        def onCommandOutputError(message):
            self.LogError(message)

        with open(jointFile, "w") as outputFile:
            with StreamFlusher(info, outputFile, lock, onCommandOutputInfo) as out:
                with StreamFlusher(error, outputFile, lock, onCommandOutputError) as err:
                    self.LogInfo("Executing '%s' in '%s'" % (command, os.getcwd()))
                    process = subprocess.Popen(command, stdout=out, stderr=err, universal_newlines=True, close_fds=True, shell=True, env=env)
                    commandReturnCode = process.wait()
                    self.LogInfo("Return code: %s" % commandReturnCode)

        with open(jointFile) as outputFile:
            with open(os.path.join(logPath, "%s.out.txt" % prefix), "w") as out:
                with open(os.path.join(logPath, "%s.err.txt" % prefix), "w") as err:
                    for line in outputFile:
                        if line.startswith(info):
                            line = line[lenInfo:]
                            out.write(line)
                        else:
                            line = line[lenError:]
                            err.write(line)

        self.TeamCityLog.BlockClose(executeBlockName)
        if commandReturnCode != 0:
            with open(os.path.join(logPath, "%s.err.txt" % prefix)) as errFile:
                self.set_info(errFile.read())
            raise CommandExecutionException("Execution of '%s' failed with code %s" % (command, commandReturnCode))

    def DumpCustomMetrics(self):
        for key, value in self.CustomMetricsDict.items():
            self.TeamCityLog.BuildStatistic(key, value)

    def GetContextKey(self, property):
        return 'sandbox.task.' + property

    def GetContextValue(self, property, default=None):
        value = self.FillPlaceholders(self.GetContextRawValue(property, default))
        return value

    def GetContextRawValue(self, property, default=None):
        return self.ctx.get(self.GetContextKey(property), default)

    def SetContextValue(self, property, value):
        self.ctx[self.GetContextKey(property)] = value

    def GetSourceDir(self, *dirs):
        if not self.ArcadiaDir:
            raise Exception("Arcadia was not checked out, pass 'svn.root' to the task context")
        return os.path.join(self.ArcadiaDir, *dirs)

    def GetBuildDir(self, *dirs):
        return os.path.realpath(self.GetSourceDir('..', 'ybuild', 'latest', '..', *dirs))

    def GetBuildRootDir(self):
        return self.GetSourceDir('..', "ybuild")

    def VerifyContext(self, property):
        if not self.ctx[self.GetContextKey(property)]:
            raise SandboxTaskUnknownError(
                "'%s' is not defined in the context. Build configuration (context) is not complete." % property)

    def VerifyCustomAction(self, action):
        self.VerifyContext(action + ".command")
        self.VerifyContext(action + ".onerror")
        validValues = ("ignore", "fail")
        value = self.GetContextValue(action + ".onerror", "fail")
        if value not in validValues:
            raise SandboxTaskUnknownError("Action '%s' has invalid value '%s'. Valid values: '%s'" % (action, value, validValues))

    def DumpTestDataList(self):
        try:
            resources = []
            marker = "golden_test_data.info.json"
            result = "test_data_description.json"
            targetDirName = "output_test_data"

            for root, dirnames, filenames in os.walk(self.TestLogsDir):
                if marker in filenames:
                    filePath = os.path.join(root, marker)
                    with open(filePath) as descriptionFile:
                        description = json.load(descriptionFile)
                        description["path"] = os.path.relpath(os.path.join(root, targetDirName), self.TestLogsDir)
                        resources.append(description)
            with open(self.GetLogPath(result), "w") as descrFile:
                json.dump(resources, descrFile)
        except Exception, e:
            self.LogError("Failed to Dump test data descriptions: %s" % e)

    def GetVaultDir(self):
        if self.VaultDir:
            return self.VaultDir

        self.VaultDir = tempfile.mkdtemp()
        data = self.GetContextValue("vaults")
        self.LogInfo("Vaults requested: {}".format(data))
        if data:
            for entry in data.split(";"):
                owner, name = entry.strip().split(":", 1)
                vaultData = self.get_vault_data(owner, name)
                filename = os.path.join(self.VaultDir, name)
                with open(filename, "w") as file:
                    file.write(vaultData)
                # rsync requires password file to be not other-accessible
                os.chmod(filename, stat.S_IRUSR | stat.S_IRGRP)
        return self.VaultDir

    def GetWorkDir(self):
        return self.abs_path()

    def GetSystemInfo(self, param):
        sysinfo = sandboxsdk.util.system_info()
        return str(sysinfo[param])

    def DumpEnvVars(self):
        self.LogInfo("Environment variables:")
        for name, value in sorted(os.environ.items(), key=lambda x: x[0]):
            self.LogInfo("  %s='%s'" % (name, value))


__Task__ = TeamcityRunner
