# -*- coding: utf-8 -*-
from __future__ import division
from __future__ import unicode_literals
from __future__ import absolute_import

import re
import logging
import requests

from six.moves.urllib.parse import urlparse, unquote, parse_qsl, urlencode, ParseResult
from json import dumps
import datetime

# import sandbox.common.types.task as ctt

from sandbox import sdk2


class YasmScreenshoter(sdk2.Task):
    supported_urls = re.compile(
        'https?://yasm\.yandex-team\.ru/(?P<yasm_path>(chart|(template/)?panel)/.*)',
        re.IGNORECASE
    )

    class Context(sdk2.Context):
        max_retries = 5

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        yasm_url = sdk2.parameters.Url('Yasm graphic url', required=True)
        with sdk2.parameters.Output:
            result_url = sdk2.parameters.Url('url with screenshot')

    def on_execute(self):
        self.Parameters.result_url = self.get_screenshot_url(self.Parameters.yasm_url)
        logging.info('Returned URL: {}'.format(self.Parameters.result_url))

    def get_yasm_screenshoter_url(self, url, width, height, static=True):
        # Unquoting URL first so we don't loose existing args
        url = unquote(url)
        # Extracting url info
        parsed_url = urlparse(url)
        # Extracting URL arguments from parsed URL
        query_params = parsed_url.query
        # Converting URL arguments to dict
        parsed_query_params = dict(parse_qsl(query_params))
        # Merging URL arguments dict with new params
        parsed_query_params.update({'width': width, 'height': height, 'static': static})

        # Bool and Dict values should be converted to json-friendly values
        parsed_query_params.update(
            {k: dumps(v) for k, v in parsed_query_params.items()
             if isinstance(v, (bool, dict))}
        )

        # Converting URL argument to proper query string
        encoded_query_params = urlencode(parsed_query_params, doseq=True)
        # Creating new parsed result object based on provided with new URL arguments.
        yasm_screenshoter_url = ParseResult(
            'https', 's.yasm.yandex-team.ru', parsed_url.path,
            parsed_url.params, encoded_query_params, parsed_url.fragment
        ).geturl()

        return yasm_screenshoter_url

    def get_screenshot_url(self, url, width=1920, height=1080,
                           timeout=datetime.timedelta(seconds=300)):
        match = self.supported_urls.match(url)
        if match:
            yasm_screenshoter_url = self.get_yasm_screenshoter_url(url, width, height)
            response = self.request_yasm_screenshoter(yasm_screenshoter_url, timeout=timeout)
        else:
            response = 'Unsupported URL'
        return response

    def request_yasm_screenshoter(self, url, timeout, retry_count=0):
        time_end = datetime.datetime.now() + timeout
        response = requests.get(url, allow_redirects=False, timeout=timeout.seconds)
        logging.info('Response: {}'.format(response))
        logging.info('Status code: {}'.format(response.status_code))
        logging.info('Text body: {}'.format(response.text))

        if response.status_code == 302 and response.headers.get('Location'):
            result_url = response.headers['Location']

        elif datetime.datetime.now() >= time_end:
            logging.error('Unsuccessful request to YASM./n Response: {}'.format(response))
            logging.fatal('Timeout making screenshot for {}'.format(url))
            result_url = 'Error making screenshot'
        elif retry_count >= self.Context.max_retries:
            logging.error('Unsuccessful request to YASM./n Response: {}'.format(response))
            logging.fatal('Max retries reached making screenshot for {}'.format(url))
            if response.status_code == 500:
                result_url = 'YASM error. Check url for errors'
            else:
                result_url = 'Error making screenshot'
        else:
            logging.error('Unsuccessful request to YASM./n Response: {}'.format(response))
            new_timeout = time_end - datetime.datetime.now()
            result_url = self.request_yasm_screenshoter(url, timeout=new_timeout,
                                                        retry_count=retry_count + 1)
        return result_url
