# -*- coding: utf-8 -*-
import os
import logging
import sys
import time
import posixpath

from datetime import datetime

import sandbox.sandboxsdk.svn as sdk_svn

from sandbox.projects.common import utils
from sandbox.projects.ab_testing import RUN_ABT_METRICS_REPORT

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.sandboxsdk.parameters import LastReleasedResource

from sandbox.projects import resource_types


def try_parse_date(path):
    if len(path) == 8:  # YYYYMMDD
        return path
    elif len(path) == 10:  # YYYY-MM-DD
        try:
            return datetime.strptime(path, '%Y-%m-%d').strftime('%Y%m%d')
        except:
            return path  # Seems it's timestamp. Leave it as string
    else:
        return None


class RunABTMetrics(SandboxTask):
    type = 'TMP_ABT_METRICS_DEV'

    class Cluster(SandboxStringParameter):
        name = 'cluster'
        description = 'YT cluster'
        default_value = 'hahn'

    class Resources(SandboxStringParameter):
        '''
        Resource files for stat_collector/stat_fetcher.
        Need files: blockstat.dict, browser.xml, geodata5.bin.
        '''
        name = 'resources_id'
        description = 'Resource files for stat_collector and stat_fetcher'

    class YtToken(SandboxStringParameter):
        '''
        Resource files for stat_collector/stat_fetcher.
        Need files: blockstat.dict, browser.xml, geodata5.bin.
        '''
        name = 'yt_token'
        description = 'yt token'

    class Package(SandboxStringParameter):
        '''
        YA_PACKAGE yandex-search-ab-testing
        '''
        name = 'package_id'
        description = 'stat_{collector, fetcher}'

    class SamplePath(SandboxStringParameter):
        '''
        Sample path or date (YYYYMMDD/YYYY-MM-DD)
        '''
        name = 'sample_path'
        description = 'Sample path or date (YYYYMMDD/YYYY-MM-DD) or timestamp'
        default_value = ''

    class VaultGroup(SandboxStringParameter):
        '''
        Vault group of yt token.
        '''
        name = 'vault_group'
        description = 'Vault group'
        default_value = 'AB-TESTING'

    class VaultName(SandboxStringParameter):
        '''
        Vault name of yt token.
        '''
        name = 'vault_name'
        description = 'Vault name'
        default_value = 'yt-token'

    class YtPool(SandboxStringParameter):
        '''
        Vault name of yt token.
        '''
        name = 'yt_pool'
        description = 'YT pool for calculation'
        default_value = ''

    class OutputPrefix(SandboxStringParameter):
        '''
        Vault name of yt token.
        '''
        name = 'output_prefix'
        description = 'YT prefix for output result'
        default_value = '//home/abt/regrabt/out/'

    class StatFetcherBinary(LastReleasedResource):
        name = 'stat_fetcher_binary'
        description = 'stat_fetcher binary'
        resource_type = resource_types.TMP_STAT_FETCHER_BINARY

    class StatCollectorBinary(LastReleasedResource):
        name = 'stat_collector_binary'
        description = 'stat_collector binary'
        resource_type = resource_types.TMP_STAT_COLLECTOR_BINARY

    class Conf(LastReleasedResource):
        name = 'sc_conf'
        description = 'conf'
        resource_type = resource_types.TMP_ABT_CONF

    input_parameters = [Cluster, Resources, Package, SamplePath, VaultGroup, VaultName, YtPool, OutputPrefix, StatFetcherBinary, StatCollectorBinary, YtToken, Conf]

    def __init__(self, *args, **kwargs):
        SandboxTask.__init__(self, *args, **kwargs)

    def on_execute(self):
        if not self.ctx['sample_path']:
            raise ValueError('Sample path not set')

        shellabt_path = sdk_svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/quality/ab_testing/scripts/shellabt/")
        sys.path.append(shellabt_path)

        import shellabt

        # TODO: Use config
        """
        cur_dir = os.getcwd()
        collector_config_path = cur_dir + '/sc_conf.txt'
        with open(collector_config_path, 'wt') as f:
            f.write('[task]\nname=\nexperiment=0\nfilter=uitype_filter:desktop')
        """
        collector_config_path = self.sync_resource(self.ctx['sc_conf'])
        logging.info("sc_conf_path:", collector_config_path)

        sample_path = utils.get_or_default(self.ctx, self.SamplePath)

        package_id = utils.get_or_default(self.ctx, self.Package)
        logging.info('Syncing resource: %d', package_id)
        package_path = self.sync_resource(package_id)

        resources_id = utils.get_or_default(self.ctx, self.Resources)
        resources = self.sync_resource(resources_id)

        conf = shellabt.FreeConfig()
        conf['server'] = utils.get_or_default(self.ctx, self.Cluster)

        date = try_parse_date(sample_path)
        if date is None:
            conf['date'] = shellabt.parse_sample_date(sample_path)
            conf['yt_prefix'] = sample_path
        else:
            conf['date'] = date

        conf['geodata'] = os.path.join(resources, 'geodata5.bin')
        conf['blockstat.dict'] = os.path.join(resources, 'blockstat.dict')
        conf['browser.xml'] = os.path.join(resources, 'browser.xml')

        extract_to = './'
        logging.info('Extracting: %s to %s', package_path, extract_to)
        # paths = shellabt.DebianYaPackagePaths(package_path, '/Berkanavt/ab_testing/bin', extract_to)
        # conf.merge(paths)
        conf['stat_fetcher'] = self.sync_resource(self.ctx['stat_fetcher_binary'])
        conf['stat_collector'] = self.sync_resource(self.ctx['stat_collector_binary'])
        logging.info("STAT_COLLECTOR_PATH", conf['stat_collector'])

        conf['yt_token'] = self.ctx['yt_token']

        yt_pool = utils.get_or_default(self.ctx, self.YtPool)
        if yt_pool:
            conf['yt_pool'] = yt_pool

        conf['output_prefix'] = utils.get_or_default(self.ctx, self.OutputPrefix)
        conf['yt_tmp'] = posixpath.join(conf['output_prefix'], 'tmp')
        conf['collector_out_table'] = '%srun_abt_metrics_%s_%s_%d' % (conf['output_prefix'], conf['date'], package_id, int(time.time()))

        stat = shellabt.StatRunner(conf)
        logging.info("STAT_ENV:", str(stat.env))
        stat.run_stat_collector(
            config_path=collector_config_path,
            target_date=conf['date'],
            result_table=conf['collector_out_table']
        )

        conf['fetcher_out_path'] = 'result.txt'

        stat.run_stat_fetcher(
            config_path=collector_config_path,
            target_date=conf['date'],
            collector_table=conf['collector_out_table'],
            result_path=conf['fetcher_out_path']
        )

        result = shellabt.StatResult.from_fetcher_file(fetcher_out_path=conf['fetcher_out_path'], fetcher_config_path=collector_config_path, stat_fetcher=conf['stat_fetcher'])

        suite = shellabt.SuiteResult(sample_path)
        suite.add(result)

        report_file = 'metrics.json'
        report_file = os.path.abspath(report_file)
        suite.to_file(report_file)
        name = 'ABT metrics resource %s prefix/date %s' % (package_id, sample_path)
        self.create_resource(name, report_file, resource_type=RUN_ABT_METRICS_REPORT, attributes={'ttl': 500})


__Task__ = RunABTMetrics
