# pylint: disable=invalid-name,missing-docstring,line-too-long

import luadump
import utils

RESERVED_MEMORY = 256 * 1024 * 1024
DEFAULT_CONFIG = {
    'server': {
        'port': 0,
        'threads': 8,
        'max_queue_size': 2048,
        'client_timeout': '30s',
        'reject_excess_connections': 1
    },
    'monitoring': {
        'answer_time_buckets': '0.1, 0.01, 0.001, 0.0001, 0.00001'
    },
    'modules': {
        'report_storable': {
            'key_builder': {
            },
            'collection': {
                'id': 'store'
            },
            'storage': {
                'arenas': 10,
                'compression': 'lz4',
                'memory_limit': '64M',
                'block_size': '1K',
                'background_storing': {
                    'threads': 8,
                    'max_queue_size': 2048,
                    'fail_on_busy': 'false'
                }
            }
        }
    }
}

ERROR_LOG_CONFIG = {
    'user_error_log': {
        'file_name': './cached-user-error.log',
        'max_log_queue_size': 65536,
    }
}

ACCESS_LOG_CONFIG = {
    'access_log': {
        'file_name': './cached-access-log',
        'max_log_queue_size': 0,
    }
}

STAT_LOG_CONFIG = {
    'stat_log': {
        'file_name': './cached-stat-log',
        'max_log_queue_size': 0,
    }
}


def _storage_memory(node_memory, reserved_memory):
    if str(reserved_memory).endswith('%'):  # str or unicode can be passed
        prc = reserved_memory[:-1]
        assert prc.isdigit(), "reserved_memory is incorrect"
        return int(node_memory * (100 - int(prc)) / 100)
    else:
        return node_memory - utils.human2bytes(reserved_memory)


def generate(
        nodes=None,
        reserved_memory=RESERVED_MEMORY,

        server_threads=8,
        server_queue_size=2048,

        storage_threads=8,
        storage_queue_size=2048,

        backend_disk_cache=None,
        disable_compression=False,

        enable_error_log=False,
        enable_access_log=False,
        enable_stat_log=False,

        max_connections=100,
        client_timeout=30,

        extra_config={},

        as_struct=False
):
    result = {}

    if not extra_config:
        extra_config = {}

    for node in nodes:
        cache_size = _storage_memory(node.memory, reserved_memory)
        if cache_size < 0:
            cache_size = 0

        disk_cache_config = {}
        if (backend_disk_cache and backend_disk_cache != '0M') or node.disk > 0:
            disk_cache_config = {
                'modules': {
                    'report_storable': {
                        'storage': {
                            'file_cache_size': backend_disk_cache or str(int(node.disk / 1024 / 1024)) + 'M',
                            'dir': 'store'
                        }
                    }
                }
            }

        compression_level = 'lz4' if not disable_compression else ''

        node_config = {
            'server': {
                'port': node.port,
                'threads': server_threads,
                'max_queue_size': server_queue_size,
                'max_connections': max_connections,
                'client_timeout': str(client_timeout) + 's',
            },
            'modules': {
                'report_storable': {
                    'storage': {
                        'memory_limit': str(int(cache_size / 1024 / 1024)) + 'M',
                        'compression': compression_level,
                        'background_storing': {
                            'threads': storage_threads,
                            'max_queue_size': storage_queue_size,
                        }
                    }
                }
            }
        }

        if enable_error_log:
            node_config = utils.mergedicts(node_config, ERROR_LOG_CONFIG)
        if enable_access_log:
            node_config = utils.mergedicts(node_config, ACCESS_LOG_CONFIG)
        if enable_stat_log:
            node_config = utils.mergedicts(node_config, STAT_LOG_CONFIG)

        config = utils.mergedicts(
            DEFAULT_CONFIG,
            node_config,
            disk_cache_config,
            extra_config
        )

        if not as_struct:
            result[node.short_hostname] = 'instance = ' + luadump.encode(config)
        else:
            result[node.short_hostname] = config

    return result
