# pylint: disable=invalid-name,missing-docstring,line-too-long

import requests
import logging
import time
import json
import re


GENCFG_GROUPS_URL = 'http://api.gencfg.yandex-team.ru/{branch}/groups'
GENCFG_GROUP_SLAVES_URL = 'http://api.gencfg.yandex-team.ru/{branch}/groups/{group}/slaves'
GENCFG_INSTANCES_URL = 'http://api.gencfg.yandex-team.ru/{branch}/searcherlookup/groups/{group}/instances'

RETRY_DELAY = 1
RETRY_AMOUNT = 3


class GencfgNode():
    def __init__(self, hostname=None, port=None, dc=None, memory=0, disk=0):
        self.hostname = hostname
        self.short_hostname = hostname.split('.', 1)[0]
        self.port = port
        self.dc = dc
        self.memory = memory
        self.disk = disk


def get_instances(branch, master_group, re_filters):
    groups = get_slave_groups(branch, master_group, re_filters)

    result = []
    for group in groups:
        result.extend(get_group_instances(branch, group))

    return result


def get_group_instances(branch, group):
    reply = query(GENCFG_INSTANCES_URL.format(branch=branch, group=group))

    assert reply, "Group '%s' (branch: '%s') is not found in GenCfg" % (group, branch)
    assert 'instances' in reply, "No 'instances' section in reply for group '%s' (branch: '%s')" % (group, branch)

    # Prepare result
    result = []
    for v in reply['instances']:
        if 'storages' not in v:
            logging.warning("'storages' section not found in instance information")

        result.append(GencfgNode(
            hostname=v['hostname'].replace('.yandex.ru', '.search.yandex.net'),
            port=v['port'],
            dc=v['dc'],
            memory=int(v['porto_limits']['memory_guarantee']),
            disk=int(v['storages']['rootfs']['size'] * 1024 * 1024 * 1024) if 'storages' in v else 0
        ))

    # Check for different DCs
    dcs = set([node.dc for node in result])
    assert len(result) > 0, "No instances in group '%s' (branch: '%s')" % (group, branch)
    assert len(dcs) == 1, "Instances of group '%s' (branch: '%s') are located in multiple DCs: %s" % (group, branch, ', '.join(dcs))

    return result


def get_slave_groups(branch, master_group, re_filters):
    reply = query(GENCFG_GROUP_SLAVES_URL.format(branch=branch, group=master_group))
    if 'slaves' not in reply:
        raise NameError("No 'slave' section in reply for group '%s' (branch: '%s')" % (master_group, branch))

    result = []
    for re_filter in re_filters:
        r = re.compile('^' + re_filter + '$')
        result.extend(filter(r.match, reply['slaves']))

    if len(result) == 0:
        raise NameError("No slave groups found by regexp: qr/^%s$/ in group '%s' (branch: '%s')" % (re_filter, master_group, branch))
    result = list(set(result))

    logging.info(
        "Slave groups for '%s' (branch: '%s') filtered by regexp %s:" % (
            master_group, branch, ', '.join(["qr/^%s$/" % v for v in re_filters])
        )
    )
    for v in result:
        logging.info(" - " + v)

    return result


def query(url):
    logging.debug("Perform query to gencfg: " + url)

    for i in xrange(1, RETRY_AMOUNT + 1):
        try:
            r = requests.get(url)
            try:
                res = json.loads(r.content)
            except Exception as e:
                raise NameError("[http code %d] unable to parse JSON: %s" % (r.status_code, e))

            assert 'error' not in res, "[http code %d] gencfg api error: %s" % (r.status_code, res['error'])

            return res
        except Exception as e:
            if i < RETRY_AMOUNT:
                logging.warning("Failed to fetch url '%s', retry %d/%d" % (url, i, RETRY_AMOUNT))
                time.sleep(RETRY_DELAY)
                continue
            else:
                raise NameError("Failed to fetch url '%s': %s" % (url, e))
