#!/usr/bin/env python2

# pylint: disable=invalid-name,missing-docstring,line-too-long
from __future__ import print_function

import os
import sys
import argparse
import json
import logging

import caas.gencfg
import caas.gencfg_mock
import caas.cached_config
import caas.proxy_config
__all__ = [caas.gencfg, caas.cached_config, caas.proxy_config]


class CaasConfigureArg():
    def __init__(self, name=None, desc=None, type=None, required=None, default=None, sandbox_value=None):
        self.name = name
        self.cli_arg = '--' + name.replace('_', '-')
        self.type = type
        self.required = required
        self.default = default
        self.desc = desc
        self.sandbox_value = sandbox_value  # Static value in sandbox


def get_argument_info():
    return [
        {
            'name': 'Common parameters',
            'args': [
                CaasConfigureArg(name='branch', desc='GenCfg branch', default='trunk', type=str),
                CaasConfigureArg(name='dest', desc='Path to dump generated configs into', type=str, sandbox_value='resources'),
            ]
        }, {
            'name': 'GenCfg parameters',
            'args': [
                CaasConfigureArg(name='parent_group', desc='GenCfg parent group for backend groups', type=str, default='ALL_DYNAMIC'),
                CaasConfigureArg(name='backend_groups_filter', desc='GenCfg backend groups regexp filter, comma-separated list', type=str, required=True),
                CaasConfigureArg(name='proxy_groups_filter', desc='GenCfg proxy groups regexp filter, comma-separated list', type=str, required=True),
            ]
        }, {
            'name': 'Backend parameters',
            'args': [
                CaasConfigureArg(name='backend_server_threads', desc='Server threads amount', default=8, type=int),
                CaasConfigureArg(name='backend_server_queue_size', desc='Server queue size', default=2048, type=int),
                CaasConfigureArg(name='backend_storage_threads', desc='Storage threads amount', default=8, type=int),
                CaasConfigureArg(name='backend_storage_queue_size', desc='Storage queue size', default=2048, type=int),
                CaasConfigureArg(name='backend_enable_error_log', desc='Enable error log', default=True, type=bool),
                CaasConfigureArg(name='backend_enable_access_log', desc='Enable access log', default=False, type=bool),
                CaasConfigureArg(name='backend_enable_stat_log', desc='Enable stat log', default=False, type=bool),
                CaasConfigureArg(name='backend_reserved_memory', desc='Reserved memory for cached process', type=str, default='256M'),
                CaasConfigureArg(name='backend_disable_compression', desc='Disable LZ4 compression', default=False, type=bool),
                CaasConfigureArg(name='backend_max_connections', desc='Max connections', default=100, type=int),
                CaasConfigureArg(name='backend_client_timeout', desc='Client timeout, sec', default=30, type=int),
                CaasConfigureArg(name='backend_disk_cache', desc='Disk cache size (default - get gencfg instance disc size)', type=str, default=None),
                CaasConfigureArg(name='backend_config', desc='Customize backend config, JSON', type=str, default=''),
            ]
        }, {
            'name': 'Proxy common parameters',
            'args': [
                CaasConfigureArg(name='proxy_quorum_reads_count', desc='Quorum reads amount', default=1, type=int),
                CaasConfigureArg(name='proxy_replication_factor', desc='Local DC replication factor', default=3, type=int),
                CaasConfigureArg(name='proxy_replication_read_factor', desc='Local DC replication read factor', default=None, type=int),
                CaasConfigureArg(name='proxy_replication_write_factor', desc='Local DC replication write factor', default=None, type=int),
                CaasConfigureArg(name='proxy_sync_writes_count', desc='Local DC sync writes count', default=2, type=int),
                CaasConfigureArg(name='proxy_max_next_nodes_count', desc='Local DC spare nodes count', default=2, type=int),
                CaasConfigureArg(name='proxy_remote_replication_factor', desc='Remote DC replication factor', default=1, type=int),
                CaasConfigureArg(name='proxy_remote_replication_read_factor', desc='Remote DC replication read factor', default=None, type=int),
                CaasConfigureArg(name='proxy_remote_replication_write_factor', desc='Remote DC replication write factor', default=None, type=int),
                CaasConfigureArg(name='proxy_remote_sync_writes_count', desc='Remote DC sync writes count', default=0, type=int),
                CaasConfigureArg(name='proxy_remote_max_next_nodes_count', desc='Remote DC spare nodes count', default=2, type=int),
                CaasConfigureArg(name='proxy_threads', desc='Proxy server threads amount', default=512, type=int),
                CaasConfigureArg(name='proxy_max_connections', desc='Proxy max inbound connections', default=100, type=int),
                CaasConfigureArg(name='proxy_max_outbound_connections', desc='Proxy max outbound connections', default=300, type=int),
                CaasConfigureArg(name='proxy_write_token', desc='Proxy write token', type=str),
                CaasConfigureArg(
                    name='proxy_answer_time_buckets',
                    desc='Answer time buckets for stats',
                    type=str,
                    default="0.1, 0.075, 0.05, 0.04, 0.03, 0.02, 0.01, 0.005, 0.001"
                ),
                CaasConfigureArg(name='proxy_config', desc='Customize proxy config, JSON', type=str, default=''),
            ]
        }, {
            'name': 'Proxy cache stats parameters',
            'args': [
                CaasConfigureArg(name='proxy_cache_stats_probes_enabled', desc='Enable TTL probes', default=True, type=bool),
                CaasConfigureArg(name='proxy_cache_stats_probes_min_depth', desc='TTL probes min depth, sec', default=30, type=int),
                CaasConfigureArg(name='proxy_cache_stats_probes_max_depth', desc='TTL probes max depth, sec', default=630, type=int),
                CaasConfigureArg(name='proxy_cache_stats_probes_amount', desc='TTL probes amount', default=40, type=int),
            ]
        }, {
            'name': 'Test options',
            'args': [
                CaasConfigureArg(name='mock_gencfg', desc='Mock HTTP queries to GenCfg', type=bool),
            ]
        }
    ]


def instances_dc(group, instances):
    if len(instances) == 0:
        raise NameError("Instances list is empty")

    dc_set = set([i['dc'] for i in instances])

    if len(dc_set) != 1:
        raise NameError(
            "All instances of group '%s' should be located in same DC, but DC list is: %s" %
            (group, ', '.join([str(s) for s in dc_set]))
        )

    return dc_set.pop()


def process(args):
    # Group filters
    if args['backend_groups_filter']:
        args['backend_groups_filter'] = [v.strip(' ') for v in args['backend_groups_filter'].split(',')]

    if args['proxy_groups_filter']:
        args['proxy_groups_filter'] = [v.strip(' ') for v in args['proxy_groups_filter'].split(',')]

    # Custom backend config
    if args['backend_config']:
        try:
            args['backend_config'] = json.loads(args['backend_config'])
        except:
            raise NameError("'backend_config' value doesn't looks like JSON string")
    else:
        args['backend_config'] = {}

    # Custom proxy config
    if args['proxy_config']:
        try:
            args['proxy_config'] = json.loads(args['proxy_config'])
        except:
            raise NameError("'proxy_config' value doesn't looks like JSON string")
    else:
        args['proxy_config'] = {}

    # Do not perform HTTP queries to GenCfg api
    if args['mock_gencfg']:
        logging.warning("Using mock GenCfg queries")
        caas.gencfg_mock.install()

    # Prepare resource dirs
    if args['dest']:
        backend_configs_path = os.path.join(args['dest'], 'cached')
        proxy_configs_path = os.path.join(args['dest'], 'caas-proxy')
        if not os.path.exists(backend_configs_path):
            os.makedirs(backend_configs_path)
        if not os.path.exists(proxy_configs_path):
            os.makedirs(proxy_configs_path)
    else:
        backend_configs_path = 'cached'
        proxy_configs_path = 'caas-proxy'

    # Generate backend configs
    logging.info("Generating backend config")
    backend_nodes = caas.gencfg.get_instances(args['branch'], args['parent_group'], args['backend_groups_filter'])

    backend_configs = caas.cached_config.generate(
        nodes=backend_nodes,
        server_threads=args['backend_server_threads'],
        server_queue_size=args['backend_server_queue_size'],
        storage_threads=args['backend_storage_threads'],
        storage_queue_size=args['backend_storage_queue_size'],
        backend_disk_cache=args['backend_disk_cache'],
        enable_error_log=args['backend_enable_error_log'],
        enable_access_log=args['backend_enable_access_log'],
        enable_stat_log=args['backend_enable_stat_log'],
        reserved_memory=args['backend_reserved_memory'],
        disable_compression=args['backend_disable_compression'],
        max_connections=args['backend_max_connections'],
        client_timeout=args['backend_client_timeout'],
        extra_config=args['backend_config']
    )

    # Write configs
    for node_shortname in sorted(backend_configs.keys()):
        filename = os.path.join(backend_configs_path, "cached-%s.conf" % node_shortname)
        if args['dest']:
            logging.info(filename)
            with open(filename, 'w') as outfile:
                outfile.write(backend_configs[node_shortname])
        else:
            logging.info("skipped: %s" % filename)

    # Generate proxy config
    logging.info('Generating proxy config')
    proxy_nodes = caas.gencfg.get_instances(args['branch'], args['parent_group'], args['proxy_groups_filter'])
    proxy_config = caas.proxy_config.generate(
        proxy_nodes=proxy_nodes,
        cache_nodes=backend_nodes,

        quorum_reads_count=args['proxy_quorum_reads_count'],
        replication_factor=args['proxy_replication_factor'],
        replication_read_factor=args['proxy_replication_read_factor'],
        replication_write_factor=args['proxy_replication_write_factor'],
        sync_writes_count=args['proxy_sync_writes_count'],
        max_next_nodes_count=args['proxy_max_next_nodes_count'],
        remote_replication_factor=args['proxy_remote_replication_factor'],
        remote_replication_read_factor=args['proxy_remote_replication_read_factor'],
        remote_replication_write_factor=args['proxy_remote_replication_write_factor'],
        remote_sync_writes_count=args['proxy_remote_sync_writes_count'],
        remote_max_next_nodes_count=args['proxy_remote_max_next_nodes_count'],
        threads=args['proxy_threads'],
        max_connections=args['proxy_max_connections'],
        max_outbound_connections=args['proxy_max_outbound_connections'],
        answer_time_buckets=args['proxy_answer_time_buckets'],
        write_token=args['proxy_write_token'],

        cache_stats_probes_enabled=args['proxy_cache_stats_probes_enabled'],
        cache_stats_probes_min_depth=args['proxy_cache_stats_probes_min_depth'],
        cache_stats_probes_max_depth=args['proxy_cache_stats_probes_max_depth'],
        cache_stats_probes_amount=args['proxy_cache_stats_probes_amount'],

        extra_config=args['proxy_config']
    )

    # Write proxy config
    proxy_config_filename = os.path.join(proxy_configs_path, "caas-proxy.conf")
    if args['dest']:
        logging.info(proxy_config_filename)
        with open(proxy_config_filename, 'w') as outfile:
            outfile.write(proxy_config)
    else:
        logging.info("skipped: %s" % proxy_config_filename)


def main():
    logging.basicConfig(format='%(levelname)s [%(funcName)s]: %(message)s', level=logging.DEBUG)

    parser = argparse.ArgumentParser(description='Generate configuration for http-proxy nodes.')
    for group in get_argument_info():
        group_parser = parser.add_argument_group(group['name'])
        for v in group['args']:
            if v.type is bool:
                group_parser.add_argument(v.cli_arg, dest=v.name, help=v.desc, action='store_true')
            else:
                group_parser.add_argument(v.cli_arg, dest=v.name, help=v.desc, type=v.type, default=v.default, required=v.required)

    args = vars(parser.parse_args(sys.argv[1:]))

    process(args)


if __name__ == "__main__":
    main()
