#!/usr/bin/python
# -*- coding: utf-8 -*-

from __future__ import print_function
import os, sys, unittest

sys.path.insert(0, os.path.join(os.path.dirname(__file__), "../.."))

import caas.cached_config
import caas.gencfg


class TestCachedConfig(unittest.TestCase):
    def test_one_node(self):
        nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', memory=512*1024*1024, port=22122),
        ]
        res = caas.cached_config.generate(nodes=nodes)

        self.assertEqual(res.keys(), ['node01'], 'node hostname not found in config')
        self.assertTrue(isinstance(res['node01'], str), 'result config isn\' a string')
        self.assertTrue(res['node01'].startswith('instance = '))

    def test_multiple_nodes(self):
        nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', memory=512*1024*1024, port=22122),
                caas.gencfg.GencfgNode(hostname='node02.yandex.net', memory=512*1024*1024, port=22122),
                caas.gencfg.GencfgNode(hostname='node03.yandex.net', memory=512*1024*1024, port=22122),
                caas.gencfg.GencfgNode(hostname='node04.yandex.net', memory=512*1024*1024, port=22122),
        ]
        res = caas.cached_config.generate(nodes=nodes)

        self.assertEqual(sorted(res.keys()), ['node01', 'node02', 'node03', 'node04'])

    def test_disk_cache(self):
        nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', memory=512*1024*1024, port=22122),
                caas.gencfg.GencfgNode(hostname='node02.yandex.net', memory=512*1024*1024, disk=4*1024*1024*1024, port=22122),
        ]
        res = caas.cached_config.generate(nodes=nodes, as_struct=True)

        self.assertNotIn('dir', res['node01']['modules']['report_storable']['storage'])
        self.assertNotIn('file_cache_size', res['node01']['modules']['report_storable']['storage'])
        self.assertEqual(res['node02']['modules']['report_storable']['storage']['dir'], 'store')
        self.assertEqual(res['node02']['modules']['report_storable']['storage']['file_cache_size'], '4096M')

    def test_custom_args(self):
        nodes = [
                caas.gencfg.GencfgNode(
                    hostname='node01.yandex.net',
                    memory=512*1024*1024,
                    port=22122,
                    disk=4*1024*1024*1024
                ),
        ]
        res = caas.cached_config.generate(
                nodes=nodes,
                reserved_memory=64*1024*1024,
                server_threads=24,
                storage_threads=8,
                as_struct=True
        )

        self.assertIn('node01', res)
        self.assertTrue(isinstance(res['node01'], dict))
        self.assertEqual(res['node01']['modules']['report_storable']['storage']['memory_limit'], str(512 - 64) + 'M')
        self.assertEqual(res['node01']['modules']['report_storable']['storage']['file_cache_size'], str(4*1024) + 'M')
        self.assertEqual(res['node01']['modules']['report_storable']['storage']['background_storing']['threads'], 8)
        self.assertEqual(res['node01']['server']['threads'], 24)
        self.assertEqual(res['node01']['server']['port'], 22122)

    def test_custom_disk_size(self):
        nodes = [
                caas.gencfg.GencfgNode(
                    hostname='node01.yandex.net',
                    memory=512*1024*1024,
                    port=22122,
                    disk=4*1024*1024*1024
                ),
        ]
        res = caas.cached_config.generate(
                nodes=nodes,
                reserved_memory=64*1024*1024,
                server_threads=24,
                storage_threads=8,
                backend_disk_cache='0M',
                as_struct=True
        )

        self.assertEqual(res['node01']['modules']['report_storable']['storage']['file_cache_size'], '0M')

    def test_reserved_memory_prc(self):
        nodes = [
                caas.gencfg.GencfgNode(
                    hostname='node01.yandex.net',
                    memory=500*1024*1024,
                    port=22122,
                    disk=4*1024*1024*1024
                ),
        ]
        res = caas.cached_config.generate(
                nodes=nodes,
                reserved_memory='10%',
                as_struct=True
        )

        self.assertIn('node01', res)
        self.assertEqual(res['node01']['modules']['report_storable']['storage']['memory_limit'], str(500 - 50) + 'M')


if __name__ == '__main__':
    unittest.main()
