#!/usr/bin/python
# -*- coding: utf-8 -*-

from __future__ import print_function
import os, sys, unittest

sys.path.insert(0, os.path.join(os.path.dirname(__file__), "../.."))

import json
import caas.gencfg
import requests

caas.gencfg.RETRY_DELAY = 0


class MockResponse():
    def __init__(self, content, ok=True):
        self.content = content
        self.ok = True
        self.status_code = 200


def _mock_requests_get(url):
    storage_info = {'rootfs': {'size': 0}}
    porto_limits_info = {'memory_guarantee': 512 * 1024 * 1024}
    mock_result = {
            '/trunk/groups/ALL_DYNAMIC/slaves': {'slaves': ['MOON_BASE', 'MARS_BASE', 'MERCURY_BASE', 'PLUTO_IS_NOT_A_PLANET']},
            '/trunk/searcherlookup/groups/MOON_BASE/instances': {
                'instances': [
                    {'dc': 'moon', 'hostname': 'moon-001.search.yandex.net', 'port': 22122, 'porto_limits': porto_limits_info, 'storages': storage_info},
                    {'dc': 'moon', 'hostname': 'moon-002.search.yandex.net', 'port': 22122, 'porto_limits': porto_limits_info, 'storages': storage_info},
                ]
            },
            '/trunk/searcherlookup/groups/PLUTO_IS_NOT_A_PLANET/instances': {
                'instances': [
                    {'dc': 'pluto', 'hostname': 'pluto-001.search.yandex.net', 'port': 22122, 'porto_limits': porto_limits_info, 'storages': storage_info},
                    {'dc': 'otulp', 'hostname': 'pluto-002.search.yandex.net', 'port': 22122, 'porto_limits': porto_limits_info, 'storages': storage_info},
                ]
            },
    }
    key = url.replace('http://api.gencfg.yandex-team.ru', '')

    if key in mock_result:
        return MockResponse(json.dumps(mock_result[key]))
    else:
        raise NameError("URL '%s' is not mocked" % url)


requests.get = _mock_requests_get


class TestCachedConfig(unittest.TestCase):
    def test_get_slave_groups(self):
        res = caas.gencfg.get_slave_groups('trunk', 'ALL_DYNAMIC', ['.*_BASE'])
        self.assertEqual(sorted(res), ['MARS_BASE', 'MERCURY_BASE', 'MOON_BASE'])

        res = caas.gencfg.get_slave_groups('trunk', 'ALL_DYNAMIC', ['MOON_BASE'])
        self.assertEqual(sorted(res), ['MOON_BASE'])

    def test_get_group_instances(self):
        res = caas.gencfg.get_group_instances('trunk', 'MOON_BASE')
        self.assertTrue(isinstance(res, list))
        for v in res:
            self.assertTrue(isinstance(v, caas.gencfg.GencfgNode))

    def test_get_group_instances_raise_on_different_dcs(self):
        self.assertRaises(AssertionError, caas.gencfg.get_group_instances, 'trunk', 'PLUTO_IS_NOT_A_PLANET')

    def test_get_instances(self):
        res = caas.gencfg.get_instances('trunk', 'ALL_DYNAMIC', ['MO.*_BASE'])
        self.assertTrue(isinstance(res, list))
        for v in res:
            self.assertTrue(isinstance(v, caas.gencfg.GencfgNode))

        self.assertEqual(res[0].hostname, 'moon-001.search.yandex.net')
        self.assertEqual(res[0].short_hostname, 'moon-001')
        self.assertEqual(res[0].port, 22122)
        self.assertEqual(res[0].dc, 'moon')
        self.assertEqual(res[0].memory, 512 * 1024 * 1024)

        self.assertEqual(res[1].hostname, 'moon-002.search.yandex.net')
        self.assertEqual(res[1].short_hostname, 'moon-002')
        self.assertEqual(res[1].port, 22122)
        self.assertEqual(res[1].dc, 'moon')
        self.assertEqual(res[1].memory, 512 * 1024 * 1024)


if __name__ == '__main__':
    unittest.main()
