#!/usr/bin/python
# -*- coding: utf-8 -*-

from __future__ import print_function
import os, sys, unittest
import json

sys.path.insert(0, os.path.join(os.path.dirname(__file__), "../.."))

import caas.proxy_config
import caas.gencfg
import caas.utils


def resolve_mock(hostname): return hostname


caas.utils.resolve = resolve_mock


class TestProxyConfig(unittest.TestCase):
    def test_simple(self):
        proxy_nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', port=80, dc='moon', memory=512*1024*1024),
                caas.gencfg.GencfgNode(hostname='node02.yandex.net', port=80, dc='moon', memory=512*1024*1024),
        ]
        cache_nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', port=22122, dc='moon', memory=512*1024*1024),
                caas.gencfg.GencfgNode(hostname='node02.yandex.net', port=22122, dc='moon', memory=512*1024*1024),
        ]
        json_res = caas.proxy_config.generate(
                proxy_nodes=proxy_nodes,
                cache_nodes=cache_nodes,
        )
        self.assertTrue(isinstance(json_res, str), 'result config isn\' a string')
        res = json.loads(json_res)

        self.assertIn('quorum-reads-count', res)
        self.assertIn('replication-read-factor', res)
        self.assertIn('replication-write-factor', res)
        self.assertIn('sync-writes-count', res)
        self.assertIn('remote-replication-read-factor', res)
        self.assertIn('remote-replication-write-factor', res)
        self.assertIn('remote-sync-writes-count', res)
        self.assertIn('max-next-nodes-count', res)
        self.assertIn('remote-max-next-nodes-count', res)
        self.assertIn('threads', res)
        self.assertIn('answer-time-buckets', res)
        self.assertIn('proxy-nodes', res)
        self.assertIn('cache-nodes', res)

        self.assertEqual(
                res['proxy-nodes'],
                [
                    {'host': 'node01.yandex.net', 'address': 'node01.yandex.net', 'port': 80, 'dc': 'moon'},
                    {'host': 'node02.yandex.net', 'address': 'node02.yandex.net', 'port': 80, 'dc': 'moon'}
                ]
        )

        self.assertEqual(
                res['cache-nodes'],
                [
                    {'host': 'node01.yandex.net', 'address': 'node01.yandex.net', 'port': 22122, 'dc': 'moon'},
                    {'host': 'node02.yandex.net', 'address': 'node02.yandex.net', 'port': 22122, 'dc': 'moon'}
                ]
        )

    def test_optional(self):
        proxy_nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', port=80, dc='moon', memory=512*1024*1024),
        ]
        cache_nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', port=22122, dc='moon', memory=512*1024*1024),
        ]
        answer_time_buckets_val = "0.1, 0.01, 0.001"
        write_token_val = "SomeSecretKey"

        json_res = caas.proxy_config.generate(
                proxy_nodes=proxy_nodes,
                cache_nodes=cache_nodes,
                answer_time_buckets=answer_time_buckets_val,
                write_token=write_token_val,
                replication_factor=3,
                remote_replication_factor=4,
        )
        res = json.loads(json_res)

        self.assertEqual(res['answer-time-buckets'], answer_time_buckets_val)
        self.assertEqual(res['write-token'], write_token_val)
        self.assertEqual(res['replication-read-factor'], 3)
        self.assertEqual(res['replication-write-factor'], 3)
        self.assertEqual(res['remote-replication-read-factor'], 4)
        self.assertEqual(res['remote-replication-write-factor'], 4)

    def test_replication_factors(self):
        proxy_nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', port=80, dc='moon', memory=512*1024*1024),
        ]
        cache_nodes = [
                caas.gencfg.GencfgNode(hostname='node01.yandex.net', port=22122, dc='moon', memory=512*1024*1024),
        ]

        json_res = caas.proxy_config.generate(
                proxy_nodes=proxy_nodes,
                cache_nodes=cache_nodes,
                replication_read_factor=3,
                replication_write_factor=4,
                remote_replication_read_factor=5,
                remote_replication_write_factor=6,
        )
        res = json.loads(json_res)

        self.assertEqual(res['replication-read-factor'], 3)
        self.assertEqual(res['replication-write-factor'], 4)
        self.assertEqual(res['remote-replication-read-factor'], 5)
        self.assertEqual(res['remote-replication-write-factor'], 6)


if __name__ == '__main__':
    unittest.main()
