# pylint: disable=invalid-name,missing-docstring,line-too-long

import json
import copy
import logging
import os

import utils

RESERVED_MEMORY = 256 * 1024 * 1024
DEFAULT_PROTO_FAMILY = 10  # IPv6
DEFAULT_CONFIG = json.loads("""
{
    "logger": {
        "level": "debug",
        "frontends": [
        ]
    },
    "options": {
        "join": true,
        "flags": 20,
        "remote": [],
        "address": [],
        "wait_timeout": 60,
        "check_timeout": 60,
        "io_thread_num": 16,
        "stall_count": 3,
        "nonblocking_io_thread_num": 16,
        "net_thread_num": 4,
        "daemon": false,
        "parallel": true,
        "auth_cookie": "",
        "bg_ionice_class": 3,
        "bg_ionice_prio": 0,
        "server_net_prio": 1,
        "client_net_prio": 6,
        "cache": {
            "size": 1073741824
        },
        "indexes_shard_count": 2,
        "monitor": {}
    },
    "backends": [
        {
            "backend_id": 1,
            "type": "blob",
            "group": 0,
            "history": "./",
            "data": "./eblob.data",
            "sync": "-1",
            "blob_flags": "158",
            "blob_size": "1G",
            "records_in_blob": "1000000",
            "periodic_timeout": 15,
            "read_only": false,
            "datasort_dir": "./"
        }
    ]
}
""")


def merge_config(current, extra):
    current = copy.deepcopy(current)
    extra = copy.deepcopy(extra)
    current_backend = current.pop('backends', [{}])[0]
    extra_backend = extra.pop('backends', [{}])[0]

    merged = utils.mergedicts(current, extra)
    merged['backends'] = [
        utils.mergedicts(current_backend, extra_backend)
    ]

    return merged


def generate(group_nodes=None, extra_config={}, proto_family=DEFAULT_PROTO_FAMILY, dest_dir=None):
    if not extra_config:
        extra_config = {}

    for group_id, nodes in group_nodes.iteritems():
        for node in nodes:
            cache_size = int(node['instance']['porto_limits']['memory_guarantee']) - RESERVED_MEMORY
            if cache_size < 0:
                cache_size = 0

            remote_addresses = ["%s:%d:%d" % (n['instance']['hostname'], n['instance']['port'], proto_family) for n in nodes if n != node]

            node_config = {
                'options': {
                    'address': ["%s:%d:%d" % (node['instance']['hostname'], n['instance']['port'], proto_family)],
                    'remote': remote_addresses,
                    'cache': {
                        'size': cache_size,
                    },
                    'auth_cookie': "caas-%04d-113edf53658e300baab660840ce3e519" % (group_id),
                },
                'backends': [
                    {
                        'group': group_id,
                    }
                ]
            }

            config = merge_config(
                DEFAULT_CONFIG,
                merge_config(
                    node_config,
                    extra_config
                )
            )

            if dest_dir:
                config_dir = os.path.join(dest_dir, 'backend')
                if not os.path.exists(config_dir):
                    os.makedirs(config_dir)

                short_hostname = node['instance']['hostname'].split('.', 1)[0]
                filename = os.path.join(config_dir, "elliptics-%s.conf" % short_hostname)
                logging.info("writing config: %s" % filename)

                with open(filename, 'w') as outfile:
                    json.dump(
                        config, outfile,
                        sort_keys=True,
                        indent=4,
                        ensure_ascii=False
                    )
            else:
                logging.info("Elliptics instance '%s:%d' config:" % (node['instance']['hostname'], node['instance']['port']))
                print(json.dumps(config, indent=4))
