#!/usr/bin/env python2

# pylint: disable=invalid-name,missing-docstring,line-too-long
from __future__ import print_function

import sys
import collections
import argparse
import json
import logging

import caas.gencfg
import caas.gencfg_mock
import caas.elliptics_config
import caas.proxy_config
__all__ = [caas.gencfg, caas.elliptics_config, caas.proxy_config]


class CaasGenCfgArg():
    def __init__(self, name=None, desc=None, type=None, required=None, default=None, sandbox_value=None):
        self.name = name
        self.cli_arg = '--' + name.replace('_', '-')
        self.type = type
        self.required = required
        self.default = default
        self.desc = desc
        self.sandbox_value = sandbox_value  # Static value in sandbox


def get_argument_info():
    return [
        {
            'name': 'Common parameters',
            'args': [
                CaasGenCfgArg(name='branch', desc='GenCfg branch', default='trunk', type=str),
                CaasGenCfgArg(name='dest', desc='Path to dump generated configs into', type=str, sandbox_value='resources'),
            ]
        },
        {
            'name': 'Backend parameters',
            'args': [
                CaasGenCfgArg(name='backend_groups', desc='GenCfg backend groups list, comma-separated', type=str, required=True),
                CaasGenCfgArg(name='local_replication_factor', desc='Replication factor for local DC', default=2, type=int),
                CaasGenCfgArg(name='local_size', desc='Relative amount of backend instances in group for storing local data', required=True, type=int),
                CaasGenCfgArg(name='remote_replication_factor', desc='Replicaton factor for remote DC', default=1, type=int),
                CaasGenCfgArg(name='remote_size', desc='Relative amount of backend instances in group for storing remote data', required=True, type=int),
                CaasGenCfgArg(name='elliptics_first_group_id', desc='First group id to enumerate elliptics groups', default=1, type=int),
                CaasGenCfgArg(name='elliptics_config', desc='Customize elliptics config, JSON', type=str, default=''),
            ]
        }, {
            'name': 'Proxy parameters',
            'args': [
                CaasGenCfgArg(name='proxy_groups', desc='GenCfg proxy groups list, comma-separated', type=str),
                CaasGenCfgArg(name='proxy_config', desc='Customize proxy config, JSON', type=str, default=''),
                CaasGenCfgArg(name='proxy_replication_dcs', desc='Customize replication DCs, format: SRC_DC:DST_DC,DST_DC; SRC_DC:DST_DC,DST_DC', type=str),
                CaasGenCfgArg(name='proxy_read_local_replica', desc='Read replica groups from local DC', type=bool),
            ]
        }, {
            'name': 'Previous backend parameters',
            'args': [
                CaasGenCfgArg(name='prev_backend_groups', desc='Previous GenCfg groups list for backend, comma-separated', type=str),
                CaasGenCfgArg(name='prev_local_replication_factor', desc='Previouds replication factor for local DC', default=2, type=int),
                CaasGenCfgArg(name='prev_local_size', desc='Previous relative amount of backend instances in group for storing local data', default=1, type=int),
                CaasGenCfgArg(name='prev_remote_replication_factor', desc='Previous replicaton factor for remote DC', default=1, type=int),
                CaasGenCfgArg(name='prev_remote_size', desc='Previous relative amount of backend instances in group for storing remote data', default=1, type=int),
                CaasGenCfgArg(name='prev_elliptics_first_group_id', desc='Previous first group id to enumerate elliptics groups', type=int),
            ]
        }, {
            'name': 'Previous proxy parameters',
            'args': [
                CaasGenCfgArg(name='prev_proxy_replication_dcs', desc='Customize previous replication DCs, format: SRC_DC:DST_DC,DST_DC; SRC_DC:DST_DC,DST_DC', type=str),
                CaasGenCfgArg(name='prev_proxy_read_local_replica', desc='Read replica groups from local DC', type=bool),
            ]
        }, {
            'name': 'Test options',
            'args': [
                CaasGenCfgArg(name='mock_gencfg', desc='Mock HTTP queries to GenCfg', type=bool),
            ]
        }
    ]


def instances_dc(group, instances):
    if len(instances) == 0:
        raise NameError("Instances list is empty")

    dc_set = set([i['dc'] for i in instances])

    if len(dc_set) != 1:
        raise NameError(
            "All instances of group '%s' should be located in same DC, but DC list is: %s" %
            (group, ', '.join([str(s) for s in dc_set]))
        )

    return dc_set.pop()


def elliptics_topology(
    groups_instances=None,
    elliptics_local_groups_amount=1,
    elliptics_local_size=1,
    elliptics_remote_groups_amount=0,
    elliptics_remote_size=0,
    elliptics_first_group_id=1
):
    elliptics_dc_first_group_id = elliptics_first_group_id
    elliptics_groups_amount = elliptics_local_groups_amount + elliptics_remote_groups_amount

    dc_groups = {}
    group_nodes = collections.defaultdict(list)  # Nodes grouped by Elliptics group id

    for gencfg_group in sorted(groups_instances.keys()):
        dc = instances_dc(gencfg_group, groups_instances[gencfg_group])

        if dc in dc_groups:
            raise NameError("Group '%s' has instances in same DC as group '%s'" % (gencfg_group, dc_groups[dc]['group']))

        dc_groups[dc] = {
            'gencfg_group': gencfg_group,
            'elliptics_groups': range(
                elliptics_dc_first_group_id,
                elliptics_dc_first_group_id + elliptics_groups_amount
            ),
            'elliptics_local_groups': range(
                elliptics_dc_first_group_id,
                elliptics_dc_first_group_id + elliptics_local_groups_amount
            ),
            'elliptics_remote_groups': range(
                elliptics_dc_first_group_id + elliptics_local_groups_amount,
                elliptics_dc_first_group_id + elliptics_groups_amount
            ),
        }

        sorted_instances = sorted(groups_instances[gencfg_group], key=lambda x: x['hostname'])
        split_i = int(
            len(sorted_instances) * elliptics_local_size * elliptics_local_groups_amount /
            (elliptics_local_size * elliptics_local_groups_amount + elliptics_remote_size * elliptics_remote_groups_amount)
        )
        instances_for_local = sorted_instances[: split_i]
        instances_for_remote = sorted_instances[split_i:]

        i = 0
        for instance in instances_for_local:
            elliptics_group_id = dc_groups[dc]['elliptics_local_groups'][i % len(dc_groups[dc]['elliptics_local_groups'])]
            group_nodes[elliptics_group_id].append({
                'instance': instance,  # gencfg info
                'gencfg_group': gencfg_group,
            })

            i = i + 1

        i = 0
        for instance in instances_for_remote:
            elliptics_group_id = dc_groups[dc]['elliptics_remote_groups'][i % len(dc_groups[dc]['elliptics_remote_groups'])]
            group_nodes[elliptics_group_id].append({
                'instance': instance,  # gencfg info
                'gencfg_group': gencfg_group,
            })
            i = i + 1

        elliptics_dc_first_group_id = elliptics_dc_first_group_id + elliptics_groups_amount

        logging.info("DC: %s, cloud group %s" % (dc, gencfg_group))
        logging.info("Elliptics groups for local data")
        for gid in dc_groups[dc]['elliptics_local_groups']:
            logging.info(" - %d: %d instances" % (gid, len(group_nodes[gid])))
        logging.info("Elliptics groups for remote data")
        for gid in dc_groups[dc]['elliptics_remote_groups']:
            logging.info(" - %d: %d instances" % (gid, len(group_nodes[gid])))

    return {
        'group_nodes': group_nodes,
        'dc_groups': dc_groups,
    }


def proxy_replication_dcs(dc_list, customized):
    """
        Get struct with replication settings.
        Custom configuration can be specified with 'custimized' option, example: "sas:man,ugrb; man:sas,ugrb; ugrb:man,sas"
    """
    result = {}
    if not customized:
        for dc in dc_list:
            result[dc] = list(set(dc_list) - set([dc]))
            logging.info("Replication DC for %s: %s" % (dc, ", ".join(result[dc])))
    else:
        customized.replace(' ', '')

        for opt in customized.split(';'):
            (src, dsts) = opt.split(':')
            assert src != '', "Bad replication setting: '%s'. Please specify replication settings in format: 'SRC:DST1[,DST2]; SRC:DST1[,DST2,...]'" % src
            assert dsts != '', "No replication DC(s) specified for DC '%s'" % src
            assert src in dc_list, "DC '%s' not in known DC list: %s" % (src, ', '.join(dc_list))

            dsts = dsts.split(',')
            for dst in dsts:
                assert dst in dc_list, "DC '%s' not in known DC list: %s" % (dst, ', '.join(dc_list))

            result[src] = dsts
            logging.info("Replication DC for %s: %s" % (src, ", ".join(result[src])))

        missing_dcs = list(set(dc_list) - set(result.keys()))
        logging.debug(len(missing_dcs))
        assert len(missing_dcs) == 0, "No replication settings given for DC(s): %s" % ', '.join(missing_dcs)

    return result


def proxy_topology(groups_instances=None, elliptics_topology=None, prev_elliptics_topology=None):
    nodes = []
    for group in groups_instances.keys():
        for instance in groups_instances[group]:
            if instance['dc'] not in elliptics_topology['dc_groups']:
                raise NameError(
                    "Instance '%s' of group '%s' has no elliptics backend instances in DC '%s'" % (
                        instance['hostname'],
                        group,
                        instance['dc']
                    )
                )

            node = {
                'instance': instance,
                'group': group,
            }

            nodes.append(node)

    return {
        'nodes': nodes,
    }


def dc_rw_groups(replica_dcs=None, prev_replica_dcs=None, dc_groups=None, prev_dc_groups=None, read_local_replica=False, prev_read_local_replica=False):
    result = {}

    for local_dc in dc_groups.keys():
        config = {
            'local_write_groups': [],
            'remote_write_groups': [],
            'read_groups_list': [],
        }

        elliptics_local_dc_groups = dc_groups[local_dc]

        # Local DC write groups
        config['local_write_groups'].extend(elliptics_local_dc_groups['elliptics_local_groups'])

        # Remote DC write groups
        for dc in replica_dcs[local_dc]:
            config['remote_write_groups'].extend(dc_groups[dc]['elliptics_remote_groups'])

        # Local DC read groups
        if read_local_replica:
            # Read all groups from local DC
            config['read_groups_list'].append(elliptics_local_dc_groups['elliptics_groups'])
        else:
            # Read only local groups from local DC
            config['read_groups_list'].append(elliptics_local_dc_groups['elliptics_local_groups'])

        # Previous local DC read groups
        if prev_dc_groups is not None and local_dc in prev_dc_groups:
            if prev_read_local_replica:
                config['read_groups_list'].append(prev_dc_groups[local_dc]['elliptics_groups'])
            else:
                config['read_groups_list'].append(prev_dc_groups[local_dc]['elliptics_local_groups'])

        # Remote DC read groups
        replica_groups = []
        for dc in replica_dcs[local_dc]:
            replica_groups.extend(dc_groups[dc]['elliptics_remote_groups'])

        if replica_groups:
            config['read_groups_list'].append(replica_groups)

        # Previous remote DC read groups
        if prev_replica_dcs is not None and local_dc in prev_replica_dcs:
            prev_replica_groups = []
            for dc in prev_replica_dcs[local_dc]:
                prev_replica_groups.extend(prev_dc_groups[dc]['elliptics_remote_groups'])

            if prev_replica_groups:
                config['read_groups_list'].append(prev_replica_groups)

        logging.info("DC %s: local write groups: %s" % (local_dc, config['local_write_groups']))
        logging.info("DC %s: remote write groups: %s" % (local_dc, config['remote_write_groups']))
        logging.info("DC %s: read groups list: %s" % (local_dc, config['read_groups_list']))

        result[local_dc] = config

    return result


def process(args):
    if args['backend_groups']:
        args['backend_groups'] = [v for v in args['backend_groups'].split(',')]

    if args['prev_backend_groups']:
        args['prev_backend_groups'] = [v for v in args['prev_backend_groups'].split(',')]

    if args['proxy_groups']:
        args['proxy_groups'] = [v for v in args['proxy_groups'].split(',')]

    if args['elliptics_config']:
        try:
            args['elliptics_config'] = json.loads(args['elliptics_config'])
        except:
            logging.error("--elliptics-config value doesn't looks like JSON string")
            raise
    else:
        args['elliptics_config'] = {}

    if args['proxy_config']:
        try:
            args['proxy_config'] = json.loads(args['proxy_config'])
        except:
            logging.error("--proxy-config value doesn't looks like JSON string")
            raise
    else:
        args['proxy_config'] = {}

    if args['mock_gencfg']:
        logging.warning("Using mock GenCfg queries")
        caas.gencfg_mock.install()

    logging.info("Generating backend config")

    et = elliptics_topology(
        groups_instances=caas.gencfg.groups_instances(args['branch'], args['backend_groups']),
        elliptics_local_groups_amount=args['local_replication_factor'],
        elliptics_remote_groups_amount=args['remote_replication_factor'],
        elliptics_local_size=args['local_size'],
        elliptics_remote_size=args['remote_size'],
        elliptics_first_group_id=args['elliptics_first_group_id']
    )

    caas.elliptics_config.generate(
        group_nodes=et['group_nodes'],
        extra_config=args['elliptics_config'],
        dest_dir=args['dest']
    )

    r_dcs = proxy_replication_dcs(et['dc_groups'].keys(), args['proxy_replication_dcs'])

    if args['proxy_groups']:
        logging.info("Generating proxy config")

        if args['prev_backend_groups']:
            prev_et = elliptics_topology(
                groups_instances=caas.gencfg.groups_instances(args['branch'], args['prev_backend_groups']),
                elliptics_local_groups_amount=args['prev_local_replication_factor'],
                elliptics_remote_groups_amount=args['prev_remote_replication_factor'],
                elliptics_local_size=args['prev_local_size'],
                elliptics_remote_size=args['prev_remote_size'],
                elliptics_first_group_id=args['prev_elliptics_first_group_id']
            )
            prev_r_dcs = proxy_replication_dcs(prev_et['dc_groups'].keys(), args['prev_proxy_replication_dcs'])
        else:
            prev_et = None
            prev_r_dcs = None

        pt = proxy_topology(
            groups_instances=caas.gencfg.groups_instances(args['branch'], args['proxy_groups']),
            elliptics_topology=et,
            prev_elliptics_topology=prev_et,
        )

        rw_groups = dc_rw_groups(
            replica_dcs=r_dcs,
            prev_replica_dcs=prev_r_dcs,
            dc_groups=et['dc_groups'],
            prev_dc_groups=prev_et['dc_groups'] if prev_et else None,
            read_local_replica=args['proxy_read_local_replica'],
            prev_read_local_replica=args['prev_proxy_read_local_replica'],
        )

        caas.proxy_config.generate(
            proxy_nodes=pt['nodes'],
            group_nodes=et['group_nodes'],
            prev_group_nodes=prev_et['group_nodes'] if prev_et else None,
            dc_rw_groups=rw_groups,
            extra_config=args['proxy_config'],
            dest_dir=args['dest'],
        )


def main():
    logging.basicConfig(format='%(levelname)s [%(funcName)s]: %(message)s', level=logging.DEBUG)

    parser = argparse.ArgumentParser(description='Generate configuration for http-proxy nodes.')
    for group in get_argument_info():
        group_parser = parser.add_argument_group(group['name'])
        for v in group['args']:
            if v.type is bool:
                group_parser.add_argument(v.cli_arg, dest=v.name, help=v.desc, action='store_true')
            else:
                group_parser.add_argument(v.cli_arg, dest=v.name, help=v.desc, type=v.type, default=v.default, required=v.required)

    args = vars(parser.parse_args(sys.argv[1:]))

    process(args)


if __name__ == "__main__":
    main()
