# coding=utf-8
import logging
import requests
import re
import ast
import json

from sandbox import sandboxsdk
from sandbox.sandboxsdk.channel import channel
from sandbox.sdk2.helpers import process
from sandbox.common.types import task as ctt
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.vcs import arc

from sandbox.projects.Travel.tasks.tools import get_target_deps, normalize_component_name

BUILD_REGEXP = re.compile("BUILD_PACKAGE:\s*(false|False|0|no|No)")
DEPLOY_REGEXP = re.compile("DEPLOY:\s*(false|False|0|no|No)")


class ComponentName(sandboxsdk.parameters.SandboxStringParameter):
    name = "component_name"
    description = "Building component name"
    required = False


class CommitMessage(sandboxsdk.parameters.SandboxStringParameter):
    name = "commit_message"
    description = "Patch's commit message"
    required = False
    multiline = True


class Committer(sandboxsdk.parameters.SandboxStringParameter):
    name = "committer"
    description = "Patch's committer"
    required = False


class Revision(sandboxsdk.parameters.SandboxStringParameter):
    name = "revision"
    description = "Patch's revision"
    required = False


class DeployEnvironment(sandboxsdk.parameters.SandboxStringParameter):
    name = "deploy_environment"
    description = "Environment for auto deploy (unstable|stable|testing). Default is testing"
    required = False


class TravelFilteringTarget(sandboxsdk.parameters.SandboxStringParameter):
    name = "travel_filtering_target"
    description = "Build is triggered only when this target deps are changed"
    required = False


class CommitPaths(sandboxsdk.parameters.SandboxStringParameter):
    name = "commit_paths"
    description = "Paths affected by commit"
    required = False


class TravelBuildBase:

    def on_create(self):
        self._report_status('Created')

    def on_enqueue(self):
        self._register_build()
        self._report_status('Enqueued')

    def set_build_skipped(self, value):
        key = 'is_build_skipped'
        self.ctx[key] = value
        channel.sandbox.set_task_context_value(self.id, key, self.ctx[key])

    def get_build_skipped(self):
        return self.ctx.get('is_build_skipped', True)

    def _has_relevant_changes(self, make_target, package_target):
        logging.info("Analyzing changes...")

        if (make_target is None and package_target is None) or (make_target is not None and package_target is not None):
            logging.info('Skipping analysis because either both or none of make_target and package_target are None')
            return True

        raw_paths = self.ctx.get(CommitPaths.name)
        if raw_paths is None:
            logging.info('Skipping analysis because raw_paths is not set')
            return True

        prefix = '/trunk/arcadia/'
        changed_paths = {x[len(prefix):] for x in ast.literal_eval(raw_paths) if x.startswith(prefix)}

        arcadia_url = self.ctx.get(ArcadiaUrl.name)
        logging.info("Arcadia url: {}".format(arcadia_url))

        with arcadia_sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=True) as arcadia_src_dir:
            dep_files = get_target_deps(arcadia_src_dir, make_target, package_target)
            if dep_files is None:
                logging.info('Skipping analysis because couldn\'t get dependency files')
                return True

            if dep_files & changed_paths:
                logging.info('Some of dependencies changed, so build is relevant')
                return True

        return False

    def should_execute(self, make_targets=None, package_targets=None):
        if self._is_build_disabled():
            self._report_status('BuildDisabled')
            self.set_build_skipped(True)
            return False

        self._report_status('Analyzing')
        if make_targets is not None and len(make_targets) != 1:
            logging.info('Expected exactly one make target')
        elif package_targets is not None and len(package_targets) != 1:
            logging.info('Expected exactly one package target')
        else:
            make_target = make_targets[0] if make_targets else None
            package_target = package_targets[0] if package_targets else None
            if not self._has_relevant_changes(make_target, package_target):
                self._report_status('BuildFiltered')
                self.set_build_skipped(True)
                return False

        self.set_build_skipped(False)
        return True

    def on_execute(self):
        self._report_status('Started')

        try:
            # Add branch to mark this revision as relevant for this component
            component = normalize_component_name(self.ctx.get(ComponentName.name))
            revision = int(self.ctx.get(Revision.name))
            branch_name = 'builds/{}/{}'.format(component, revision)
            logging.info("Add branch {} to mark revision {} as relevant for component {}".format(branch_name, revision,
                                                                                                 component))
            self._create_branch(branch_name, revision)
        except Exception:
            logging.warning("Failed to mark revision", exc_info=True)

    def was_executed(self):
        return not self.get_build_skipped()

    def on_success(self):
        if not self.get_build_skipped():
            self._report_status('Succeed')
            self._release_self()

    def on_failure(self):
        self._report_status('Failed')

    def on_timeout(self):
        self._report_status('Failed')

    def on_break(self):
        self._report_status('Failed')

    def before_release(self, additional_parameters):
        release_status = additional_parameters['release_status']
        if release_status == ctt.ReleaseStatus.UNSTABLE:
            self._report_status('ReleasingToUnstable')
        if release_status == ctt.ReleaseStatus.TESTING:
            self._report_status('ReleasingToTesting')
        if release_status == ctt.ReleaseStatus.STABLE:
            self._report_status('ReleasingToStable')

    def after_release(self, additional_parameters):
        release_status = additional_parameters['release_status']
        releaser = additional_parameters['releaser']
        if release_status == ctt.ReleaseStatus.UNSTABLE:
            self._report_status('ReleasedToUnstable')
        if release_status == ctt.ReleaseStatus.TESTING:
            self._report_status('ReleasedToTesting')
        if release_status == ctt.ReleaseStatus.STABLE:
            self._handle_stable_release(releaser)
            self._report_status('ReleasedToStable')

    def _release_self(self):
        if self._is_deploy_disabled():
            self._report_status('ReleaseDisabled')
            return
        self.on_release(dict(
            releaser=self.author,
            release_status=self._get_release_environment(),
            release_subject=self._get_release_title(),
            email_notifications=dict(to=[], cc=[]),
            release_comments=self.ctx.get(CommitMessage.name),
        ))

    def _get_release_title(self):
        message = self.ctx.get(CommitMessage.name)
        if '\n' in message:
            return message[:message.find('\n')].strip()
        if len(message) > 150:
            return message[:150] + "..."
        return message

    def _register_build(self):
        data = {
            'commit_message': self.ctx.get(CommitMessage.name),
            'committer': self.ctx.get(Committer.name),
            'revision': self.ctx.get(Revision.name),
            'environment': self._get_release_environment(),
        }
        self._send_notification_info('register-build', data)

    def _report_status(self, status):
        data = {
            'revision': self.ctx.get(Revision.name),
            'component_name': self.ctx.get(ComponentName.name),
            'status': status,
            'task_id': self.id,
        }
        self._send_notification_info('report-status', data)

    def _report_stable_release(self, revision_infos_to_report, releaser):
        data = {
            'revision': self.ctx.get(Revision.name),
            'component_name': self.ctx.get(ComponentName.name),
            'revisions': revision_infos_to_report,
            'task_id': self.id,
            'releaser': releaser,
            'env': 'stable'
        }
        self._send_notification_info('report-release', data)

    def _send_notification_info(self, type, data):
        try:
            url = 'http://travel-slack-forwarder.yandex.net:9098/build-notifications/{}'.format(type)
            logging.info("Sending notification info to {}. data: {}".format(url, data))
            resp = requests.post(url, json=data, timeout=10)
            if not resp.ok:
                logging.warning("Failed to send notification info. status_code: {}, response: {}".format(resp.status_code, resp.content))
        except Exception:
            logging.warning("Failed to send notification", exc_info=True)

    def _get_release_environment(self):
        mapping = {
            'testing': ctt.ReleaseStatus.TESTING,
            'stable': ctt.ReleaseStatus.STABLE,
            'unstable': ctt.ReleaseStatus.UNSTABLE
        }
        return mapping.get(self.ctx.get(DeployEnvironment.name), ctt.ReleaseStatus.TESTING)

    def _is_build_disabled(self):
        return BUILD_REGEXP.search(self.ctx.get(CommitMessage.name)) is not None

    def _is_deploy_disabled(self):
        return DEPLOY_REGEXP.search(self.ctx.get(CommitMessage.name)) is not None

    def _arc_list_branches(self, mount_point, prefix):
        logging.info("Getting branches")
        try:
            out = process.subprocess.check_output([arc.Arc()._client_bin, "branch", "--all", "--json"],
                                                  cwd=mount_point)
            return [x['name'] for x in json.loads(out.strip()) if prefix is None or x['name'].startswith(prefix)]
        except process.subprocess.CalledProcessError as e:
            logging.error(str(e))
            raise arc.ArcCommandFailed("unable to list branches")

    def _arc_get_revision_info(self, mount_point, revisions):
        logging.info("Getting revision info")
        try:
            str_revisions = ['r{}'.format(x) for x in revisions]
            out = process.subprocess.check_output(
                [arc.Arc()._client_bin, "show", "--name-only", "--json"] + str_revisions, cwd=mount_point)
            result = []
            for item in json.loads(out.strip()):
                commit = item['commits'][0]
                result.append({
                    'author': commit['author'],
                    'date': commit['date'],
                    'revision': commit['revision'],
                    'message': commit['message'],
                })
            return result
        except process.subprocess.CalledProcessError as e:
            logging.error(str(e))
            raise arc.ArcCommandFailed("unable to list branches")

    def _arc_delete_branch(self, mount_point, branch_name):
        logging.info("Deleting branch {}".format(branch_name))
        try:
            command = [arc.Arc()._client_bin, "push", "-d", branch_name]
            process.subprocess.check_call(command, cwd=mount_point)
        except process.subprocess.CalledProcessError as e:
            logging.error(str(e))
            raise arc.ArcCommandFailed("unable to delete branch {}".format(branch_name))

    def _create_branch(self, branch_name, revision):
        with arc.Arc().mount_path(None, 'r{}'.format(revision), fetch_all=True) as arcadia_src_dir:
            self._do_create_branch(arcadia_src_dir, branch_name)

    def _do_create_branch(self, arcadia_src_dir, branch_name):
        arc.Arc().checkout(arcadia_src_dir, branch=branch_name, create_branch=True)
        arc.Arc().push(arcadia_src_dir, 'users/robot-travel-prod/{}'.format(branch_name))

    def _handle_stable_release(self, releaser):
        try:
            component = self.ctx.get(ComponentName.name)
            revision = int(self.ctx.get(Revision.name))
            normalized_component = normalize_component_name(component)
            branch_name = 'releases/stable/{}/{}'.format(normalized_component, revision)
            logging.info("Add branch {} to mark revision {} as released for component {}".format(branch_name, revision,
                                                                                                 component))
            prefix = 'arcadia/users/robot-travel-prod'
            with arc.Arc().mount_path(None, 'r{}'.format(revision), fetch_all=True) as arcadia_src_dir:
                branches = self._arc_list_branches(arcadia_src_dir, prefix)
                self._do_create_branch(arcadia_src_dir, branch_name)
                branches = [x[len(prefix):] for x in branches]
                logging.info('Found branches: {}'.format(branches))
                release_revisions = [int(x.split('/')[-1]) for x in branches if x.startswith('/releases/stable/{}/'.format(normalized_component))]
                logging.info('Release revisions: {}'.format(release_revisions))
                if len(release_revisions) > 0:
                    last_release_revision = max(release_revisions)
                    build_revisions = [int(x.split('/')[-1]) for x in branches if x.startswith('/builds/{}/'.format(normalized_component))]
                    revisions_to_report = [x for x in build_revisions if last_release_revision < x <= int(revision)]
                    revisions_to_delete = [x for x in build_revisions if x <= last_release_revision]
                    logging.info('Revisions to report: {}'.format(revisions_to_report))
                    logging.info('Revisions to delete: {}'.format(revisions_to_delete))
                    revision_infos_to_report = self._arc_get_revision_info(arcadia_src_dir, revisions_to_report)
                    self._report_stable_release(revision_infos_to_report, releaser)
                    for revision in revisions_to_delete:
                        self._arc_delete_branch(arcadia_src_dir, 'users/robot-travel-prod/builds/{}/{}'.format(normalized_component, revision))

        except Exception:
            logging.warning("Failed to mark revision", exc_info=True)
