import logging
import tempfile

from sandbox import sandboxsdk
from sandbox import sdk2

import sandbox.common.types.misc as ctm
from sandbox.projects.Travel import resources
from sandbox.sdk2.helpers import subprocess as sp

TEST_SCRIPTS = [
    "expedia_acceptance_test.py", "trust_invoice_acceptance_test.py"
]


class TravelOrdersAppAcceptanceRunner(sdk2.Task):
    """ Orders app run acceptance tests runner"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024  # 10GB
        ram = 4 * 1024  # 4GB
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        vault_oauth_token = sdk2.parameters.Vault("Vault oauth token", default_value='ROBOT_TRAVEL_DEV_VAULT_OAUTH_TOKEN',
                                                  required=True)
        orders_app_resource = sdk2.parameters.Resource("Orders app resource", description="Orders app resource",
                                                       resource_type=resources.TRAVEL_ORDERS_APP_BINARY,
                                                       required=True)

    class Context(sdk2.Context):
        test_scripts = dict()

    def on_execute(self):
        logging.info("Pre-release build 2")
        with self.memoize_stage.run_acceptance_tests:
            self._run_acceptance_tests()

    def _run_acceptance_tests(self):
        logging.info("Resource status: {}".format(self.Parameters.orders_app_resource))

        app_data = sdk2.ResourceData(self.Parameters.orders_app_resource)  # synchronizing resource data on disk
        app_path = app_data.path

        orders_app_dir = tempfile.mkdtemp(suffix="orders-app")
        java_gc_log_dir = tempfile.mkdtemp(suffix="java-gc-log")
        self._run_required_subprocess(["tar", "-xzf", str(app_path), "--directory", orders_app_dir])

        for script in TEST_SCRIPTS:
            self._run_test_subprocess(orders_app_dir, java_gc_log_dir, script)

        for _, value in self.Context.test_scripts.iteritems():
            if value != 0:
                raise sandboxsdk.errors.SandboxTaskFailureError("One or more acceptance tests failed")

    def _run_required_subprocess(self, cmds):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("subprocess")) as pl:
            proc = sp.Popen(cmds, stdout=pl.stdout, stderr=sp.STDOUT)
            proc.wait()
            if proc.returncode != 0:
                raise sandboxsdk.errors.SandboxTaskFailureError(
                    "Subprocess with cmds: {} failed with exit code {}".format(cmds, proc.returncode))

    def _run_test_subprocess(self, orders_app_dir, java_gc_log_dir, script):
        vault_oauth_token = self.Parameters.vault_oauth_token.data()
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("acceptance-test")) as pl:
            proc = sp.Popen("bin/{}".format(script),
                            cwd="{}/app/acceptance".format(orders_app_dir),
                            stdout=pl.stdout,
                            env=dict(JAVA_BINARY="{}/app/acceptance/jdk/bin/java".format(orders_app_dir),
                                     BASE_LOG_DIR=java_gc_log_dir,
                                     VAULT_OAUTH_TOKEN=vault_oauth_token),
                            stderr=sp.STDOUT)
            proc.wait()
            self.Context.test_scripts[script] = proc.returncode
