import logging
import tempfile

import sandbox.common.types.task as ctt
from sandbox import sandboxsdk
from sandbox import sdk2
from sandbox.common.utils import chain
from sandbox.sdk2.helpers import subprocess as sp

TEST_SCRIPTS = [
    "expedia_acceptance_test.py", "trust_invoice_acceptance_test.py"
]


class TravelOrdersAppRelease(sdk2.Task):
    """ Package orders app and run acceptance tests """

    class Parameters(sdk2.Task.Parameters):
        checkout_from_url = sdk2.parameters.String("Checkout", description="Checkout from",
                                                   default="arcadia:/arc/trunk/arcadia")
        vault_oauth_token_name = sdk2.parameters.String("Vault oauth token name", description="Vault oauth token name",
                                                        default="ROBOT_TRAVEL_DEV_VAULT_OAUTH_TOKEN")
        use_fuse_api = sdk2.parameters.Bool("Use fuse api", default=True)
        use_selective_checkout = sdk2.parameters.Bool("Selective checkout", default=True)
        selective_checkout_mode = sdk2.parameters.String("Selective checkout mode",
                                                         description="Selective checkout mode",
                                                         choices=[
                                                             ("auto", "auto"), ("manual", "manual"),
                                                             ("default", "default")
                                                         ], default="manual", required=True)

    class Context(sdk2.Context):
        package_task_id = None
        test_scripts = dict()

    def on_execute(self):
        logging.info("Pre-release build 2")
        with self.memoize_stage.build_package:
            self._build_package()
        with self.memoize_stage.run_acceptance_tests:
            self._run_acceptance_tests()

    def _build_package(self):
        task_class = sdk2.Task["YA_PACKAGE"]
        package_task = task_class(self,
                                  owner=self.Parameters.owner,
                                  priority=self.Parameters.priority,
                                  description="Pre-release build",
                                  checkout_arcadia_from_url=self.Parameters.checkout_from_url,
                                  use_aapi_fuse=self.Parameters.use_fuse_api,
                                  checkout=self.Parameters.use_selective_checkout,
                                  checkout_mode=self.Parameters.selective_checkout_mode,
                                  package_type="tarball",
                                  packages="travel/orders/app/package/package.json",
                                  run_tests=True,
                                  resource_type="TRAVEL_ORDERS_APP_BINARY",
                                  publish_package=False,
                                  use_new_format=True,
                                  run_long_tests=True)
        package_task.Requirements.disk_space = self.Requirements.disk_space
        package_task.save()
        self.Context.package_task_id = package_task.id
        package_task.enqueue()
        raise sdk2.WaitTask(package_task, chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK), wait_all=True,
                            timeout=1200)

    def _run_acceptance_tests(self):
        package_task_status = sdk2.Task["YA_PACKAGE"].find(id=self.Context.package_task_id).first().status
        logging.info("Child package task status: {}".format(package_task_status))

        if package_task_status != ctt.Status.SUCCESS:
            raise sandboxsdk.errors.SandboxTaskFailureError("Child package task status is not SUCCESS")

        # travel_orders_app_binary_resource = sdk2.Resource['TRAVEL_ORDERS_APP_BINARY'].find(id=81).first()
        travel_orders_app_binary_resource = sdk2.Resource["TRAVEL_ORDERS_APP_BINARY"].find(
            task_id=self.Context.package_task_id, ).first()
        if not travel_orders_app_binary_resource:
            raise sandboxsdk.errors.SandboxTaskFailureError("Packaged app binary not found")

        app_data = sdk2.ResourceData(travel_orders_app_binary_resource)  # synchronizing resource data on disk
        app_path = app_data.path

        orders_app_dir = tempfile.mkdtemp(suffix="orders-app")
        java_gc_log_dir = tempfile.mkdtemp(suffix="java-gc-log")
        self._run_required_subprocess(["tar", "-xzf", str(app_path), "--directory", orders_app_dir])

        for script in TEST_SCRIPTS:
            self._run_test_subprocess(orders_app_dir, java_gc_log_dir, script)

        for _, value in self.Context.test_scripts.iteritems():
            if value != 0:
                raise sandboxsdk.errors.SandboxTaskFailureError("One or more acceptance tests failed")

    def _run_required_subprocess(self, cmds):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("subprocess")) as pl:
            proc = sp.Popen(cmds, stdout=pl.stdout, stderr=sp.STDOUT)
            proc.wait()
            if proc.returncode != 0:
                raise sandboxsdk.errors.SandboxTaskFailureError(
                    "Subprocess with cmds: {} failed with exit code {}".format(cmds, proc.returncode))

    def _run_test_subprocess(self, orders_app_dir, java_gc_log_dir, script):
        vault_oauth_token = sdk2.Vault.data(self.Parameters.vault_oauth_token_name)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("acceptance-test")) as pl:
            proc = sp.Popen("bin/{}".format(script),
                            cwd="{}/acceptance".format(orders_app_dir),
                            stdout=pl.stdout,
                            env=dict(JAVA_BINARY="{}/acceptance/jdk/bin/java".format(orders_app_dir),
                                     BASE_LOG_DIR=java_gc_log_dir,
                                     VAULT_OAUTH_TOKEN=vault_oauth_token),
                            stderr=sp.STDOUT)
            proc.wait()
            self.Context.test_scripts[script] = proc.returncode
