# coding=utf-8
from __future__ import unicode_literals

import logging
import yaml
import os.path
from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.types import resource
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.avia.lib.resource_update_detector import act_only_on_resource_update

logger = logging.getLogger(__name__)


@act_only_on_resource_update(
    enable_by_parameter='act_only_on_resource_update',
    resource_type_parameter='resource_type',
    context_storage_name='_last_processed_resource_version',
)
class TravelUpdateDynamicResources(binary_task.LastBinaryTaskRelease, sdk2.Task):
    OUTPUT_PATH = './output'

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)
        act_only_on_resource_update = sdk2.parameters.Bool('Act only on resource update', default=False)
        config_path = sdk2.parameters.SvnUrl(
            'Path to dynamic resource config.yaml in Arcadia',
            required=True,
            description='e.g. {}'.format(Arcadia.trunk_url('travel/notifier/config/dynamic_resources/dicts.yaml'))
        )
        yp_address = sdk2.parameters.String('YP address', default_value='xdc.yp.yandex.net:8090', required=True)
        robot_login = sdk2.parameters.String(
            'Robot login',
            description='login is used to run ya tool dctl ... (e.g. robot-avia-api-pilot)',
            required=True,
        )
        dctl_token = sdk2.parameters.YavSecret(
            'DCTL_YP_TOKEN',
            description='e.g. sec-01exyrpzw0wmecz2aeph888fpj',
            required=True,
        )
        ya_token = sdk2.parameters.YavSecret(
            'YA_TOKEN',
            description='e.g. sec-01ey4ehe7m8tnp56xm64v603vh',
            required=True,
        )
        resource_type = sdk2.parameters.String(
            'Sandbox-resource type',
            required=True,
            default_value='TRAVEL_RASP_COMMON_DICTS_BUNDLE',
        )
        deploy_project = sdk2.parameters.String('YDeploy project', description='e.g. travel-notifier', required=True)
        deploy_unit = sdk2.parameters.String('YDeploy deploy unit', description='e.g. api', required=True)
        deploy_stages = sdk2.parameters.List(
            'Deploy stages',
            description='Resources will be updated only for specified stages. All stages will be updated if the list is empty. e.g. travel-notifier-unstable, travel-notifier-testing',
            default=[],
        )

    def on_execute(self):
        super(TravelUpdateDynamicResources, self).on_execute()

        from yp.client import YpClient

        yp_client = YpClient(self.Parameters.yp_address, config={'token': self.Parameters.dctl_token.data()['token']})

        output_dir = self._get_output_dir()

        with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia_root:
            ya_path = os.path.join(arcadia_root, 'ya')
            for filename, content, cluster in self._generate_configs(yp_client):
                try:
                    config_path = self._save_config_file(output_dir, filename, content)
                    self._update_dynamic_resource(config_path.absolute().as_posix(), ya_path, cluster)
                except:
                    logger.exception('failed to update dynamic resource')

    def _generate_configs(self, yp_client):
        last_resource = sdk2.Resource.find(type=self.Parameters.resource_type, state=resource.State.READY, order='-id', limit=1).first()
        if not last_resource:
            raise Exception('failed to get resource of type %s', self.Parameters.resource_type)
        logger.info('config: %s', self.Parameters.config_path)
        template = Arcadia.cat(self.Parameters.config_path)
        for cluster, pod_set_id in self._get_targets(yp_client):
            config = yaml.load(template)
            stage_id = pod_set_id.split('.')[0]
            if self.Parameters.deploy_stages and stage_id not in self.Parameters.deploy_stages:
                continue
            config['meta']['id'] = config['meta']['id'].format(pod_set_id)
            config['meta']['pod_set_id'] = pod_set_id
            config['spec']['deploy_groups'][0]['urls'] = [
                last_resource.skynet_id.encode('utf8'),
                'sbr:{}'.format(last_resource.id).encode('utf8'),
            ]
            yield '{}-{}-{}'.format(pod_set_id, cluster, os.path.split(self.Parameters.config_path)[-1]), config, cluster

    def _get_targets(self, yp_client):
        stages = yp_client.select_objects(
            'stage'.encode('utf8'),
            selectors=['/status/deploy_units/{}/replica_set/cluster_statuses'.format(self.Parameters.deploy_unit)],
            filter="[/meta/project_id]=\"{}\"".format(self.Parameters.deploy_project),
        )
        for stage in stages:
            for cluster in stage[0]:
                yield str(cluster), str(stage[0][cluster]['endpoint_set_id'])

    @staticmethod
    def _save_config_file(output_dir, filename, content):
        filename = output_dir / filename
        logger.info('save config file:')
        logger.info('\n' + yaml.dump(content))
        with open(str(filename), 'w') as f:
            yaml.dump(content, f)
        return filename

    def _update_dynamic_resource(self, config_path, ya_path, cluster):
        environment_vars = {
            'DCTL_YP_TOKEN': self.Parameters.dctl_token.data()['token'],
            'YA_TOKEN': self.Parameters.ya_token.data()['token'],
            'LOGNAME': self.Parameters.robot_login,
        }
        command = '{} tool dctl put dr --cluster {} {}'.format(ya_path, cluster, config_path)
        logger.info('run update command: %s', command)
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            return_code = sdk2.helpers.subprocess.Popen(
                command,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=environment_vars,
            ).wait()
            if return_code != 0:
                error_message = 'Child ya tool dctl process has returned non-zero exit code: {}'.format(return_code)
                logger.error(error_message)
                raise Exception(error_message)

    def _get_output_dir(self):
        output_dir = Path(self.OUTPUT_PATH)
        output_dir.mkdir(parents=True, exist_ok=True)
        logger.info('Created directory {}'.format(output_dir.absolute()))
        return output_dir
