# coding=utf-8
import logging
import os
import json
import glob
import re


import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox import sandboxsdk


def run_ya_command(source_root, args, log_prefix):
    try:
        get_deps_cmd = [
            arcadia_sdk._python(),
            arcadia_sdk._get_ya_tool(source_root),
        ] + args

        pr = sandboxsdk.process.run_process(get_deps_cmd, log_prefix=log_prefix, outputs_to_one_file=False)
        sandboxsdk.process.check_process_timeout(pr, 1200, timeout_sleep=1)  # 20m
        sandboxsdk.process.check_process_return_code(pr)
        with open(pr.stdout_path) as f:
            return json.load(f)

    except Exception:
        logging.error("Failed to run ya make", exc_info=True)


def get_ya_make_deps(source_root, target_path):
    try:
        modules = run_ya_command(source_root, ['dump', 'dir-graph', '--dump-deps', os.path.join(source_root, target_path)], 'deps_ya_make_graph')
        return {x + '/ya.make' for x in modules}
    except Exception:
        logging.error("Failed to build ya make deps graph", exc_info=True)


def get_target_deps_graph(source_root, target_path):
    try:
        graph = run_ya_command(source_root, ['make', '-tt', '-j0', '-k', '--dump-json-graph', '--strict-inputs', os.path.join(source_root, target_path)], 'deps_graph')
        inputs = sum([x['inputs'] for x in graph['graph']], [])
        files = set()
        for prefix in ['$(SOURCE_ROOT)/', '$S/']:
            files |= {x[len(prefix):] for x in inputs if x.startswith(prefix)}

        return files
    except Exception:
        logging.error("Failed to build deps graph", exc_info=True)


def get_all_ya_make_deps(source_root, target_path):
    first = get_target_deps_graph(source_root, target_path)
    second = get_ya_make_deps(source_root, target_path)
    return first | second


def get_all_ya_package_deps(source_root, target_path):
    with open(os.path.join(source_root, target_path)) as f:
        pkg = json.load(f)
    files = {target_path}
    for target in pkg['build']['targets']:
        files |= get_target_deps_graph(source_root, target)
        files |= get_ya_make_deps(source_root, target)
    for item in pkg['data']:
        source = item['source']
        if source['type'] == 'ARCADIA':
            abspaths = []
            if 'files' in source:
                for file in source['files']:
                    abspaths += glob.glob(os.path.join(source_root, source['path'], file))
            else:
                curr_path = os.path.join(source_root, source['path'])
                if os.path.isfile(curr_path):
                    abspaths.append(curr_path)
                else:
                    for (dirpath, dirnames, filenames) in os.walk(curr_path):
                        abspaths += [os.path.join(dirpath, x) for x in filenames]
            files |= set([os.path.relpath(x, source_root) for x in abspaths])
    return files


def get_target_deps(source_root, make_target, package_target):
    try:
        if make_target:
            return get_all_ya_make_deps(source_root, make_target)
        if package_target:
            return get_all_ya_package_deps(source_root, package_target)
    except Exception:
        logging.error("Failed to get target deps", exc_info=True)


def normalize_component_name(component_name):
    return re.sub('\W', '-', component_name).lower()
