import logging
import requests
import time

from sandbox import common
from sandbox.projects.common import solomon
from sandbox.sdk2 import yav


class TaskTrackerConfig(object):
    use_vault_filter = False

    enable_solomon = False
    solomon_track_start = False

    solomon_secret_token = ''
    solomon_secret_key = ''
    solomon_secret_version = ''

    solomon_project = ''
    solomon_cluster = ''
    solomon_service = ''
    solomon_common_labels = {}
    solomon_sensor_delta_tail_seconds = 2

    solomon_token = None
    solomon_shard = None
    source = None
    host = None

    REQUIRED_PARAMS = [
        'solomon_secret_token',
        'solomon_secret_key',

        'solomon_project',
        'solomon_cluster',
        'solomon_service',

        'source',
        'host',
    ]

    _configured = False

    def apply(self):
        if self._configured:
            return

        if not self.enable_solomon:
            return

        if not self._check_required_params():
            raise common.errors.TaskFailure(
                'TaskTrackerConfig can not find required parameters: {}'.format(self.REQUIRED_PARAMS))

        secret_version = self.solomon_secret_version if self.solomon_secret_version else None
        self.solomon_token = self._load_secret(self.solomon_secret_token, self.solomon_secret_key,
                                               secret_version, self.use_vault_filter)

        self.solomon_shard = {
            'project': self.solomon_project,
            'cluster': self.solomon_cluster,
            'service': self.solomon_service,
        }

        if self.solomon_common_labels is None:
            self.solomon_common_labels = {}
        self.solomon_common_labels.update({
            'host': self.host,
            'source': self.source,
        })

        self._configured = True

    def _check_required_params(self):
        return all(getattr(self, param) for param in self.REQUIRED_PARAMS)

    def _load_secret(self, secret_id, key, version=None, with_filter=False):
        secret_data = yav.Secret(secret_id, version).data()
        value = secret_data[key]
        if with_filter:
            vault_filter = common.log.VaultFilter.filter_from_logger(logging.getLogger())
            if vault_filter:
                vault_filter.add_record(secret_id, value)
            else:
                raise RuntimeError('Failed to setup VaultFilter')

        return value


class CommonTaskTrackerMixin(object):
    _solomon_configured = False

    _solomon_status_ok = 'OK'
    _solomon_status_fail = 'CRIT'
    _solomon_status_running = 'RUNNING'
    _solomon_prefix = 'sandbox'

    task_tracker_config = TaskTrackerConfig()

    def fill_task_tracker_config(self):
        raise NotImplementedError('Implement config filling in fill_task_tracker_config method')

    def _solomon_configure(self):
        if self._solomon_configured:
            return

        self.fill_task_tracker_config()
        self.task_tracker_config.apply()

        self._solomon_configured = True
        logging.info('CommonTaskTrackerMixin configure done')

    def on_prepare(self):
        self._track_start()

    def _make_sensor(self, labels, status, value, ts):
        sensor_labels = {
            'sensor': status,
        }
        if labels:
            sensor_labels.update(labels)
        return {
            'labels': sensor_labels,
            'ts': ts,
            'value': value
        }

    def _available(self):
        self._solomon_configure()
        if not self._solomon_configured:
            raise common.errors.TaskFailure('CommonTaskTrackerMixin not configured')

        if not self.task_tracker_config.enable_solomon:
            logging.info('Unable to push: solomon not enabled')
            return False

        return True

    def _track_start(self):
        logging.info('Try to track start')
        if not self._available():
            return

        self.Context.solomon_start_point = int(time.time())
        self.Context.solomon_make_start_point = True
        self.Context.save()
        sensors = [self._make_sensor(None, self._solomon_status_running, 1, self.Context.solomon_start_point)]
        self._push(sensors)

    def _prepare_and_push(self, status, labels=None):
        if not self._available():
            return

        now = int(time.time())
        sensors = [self._make_sensor(labels, status, 1, now),
                   self._make_sensor(labels, status, 0, now + self.task_tracker_config.solomon_sensor_delta_tail_seconds)]

        if self.Context.solomon_make_start_point:
            sensors.extend([
                self._make_sensor(labels, status, 1, self.Context.solomon_start_point),
                self._make_sensor(None, self._solomon_status_running, 1, now),
                self._make_sensor(None, self._solomon_status_running, 0,
                                  now + self.task_tracker_config.solomon_sensor_delta_tail_seconds),
            ])
        self._push(sensors)

    def _push(self, sensors):
        logging.info('Try to push sensors to %s', self.task_tracker_config.solomon_shard)
        try:
            solomon.push_to_solomon_v2(self.task_tracker_config.solomon_token, self.task_tracker_config.solomon_shard,
                                       sensors, self.task_tracker_config.solomon_common_labels)
        except requests.exceptions.HTTPError as err:
            logging.error('Error pushing sensors to solomon: %s', err)

    def on_success(self, prev_status):
        logging.info('Process on_success handler')
        self._prepare_and_push(self._solomon_status_ok)

        super(CommonTaskTrackerMixin, self).on_success(prev_status)

    def on_failure(self, prev_status):
        logging.info('Process on_failure handler')
        labels = {'description': 'error'}
        self._prepare_and_push(self._solomon_status_fail, labels)

        super(CommonTaskTrackerMixin, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        logging.info('Process on_break handler')
        labels = {'description': 'break'}
        self._prepare_and_push(self._solomon_status_fail, labels)

        super(CommonTaskTrackerMixin, self).on_break(prev_status, status)
