# -*- coding: utf-8 -*-

import logging
import os

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.parameters import SandboxStringParameter, LastReleasedResource, SandboxArcadiaUrlParameter

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel

from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects import resource_types

from sandbox.projects.common import apihelpers


Binaries = [
    (resource_types.TV_SNIP_DATA_BUILD_GMAKE, 'gmake_id'),
    (resource_types.TV_SNIP_DATA_BUILD_CLASS_DATE_TGZ, 'class_date_id'),
]


class ArcadiaTV(SandboxArcadiaUrlParameter):
    name = 'arcadia_tv'
    description = 'Arcadia path to tv'
    default_value = Arcadia.trunk_url() + '/extsearch/wizards/tv'
    required = True


class BinariesResources(LastReleasedResource):
    name = 'sandbox_resource'
    description = 'Executables and other resources'
    required = True


class TvExportProxy(SandboxStringParameter):
    name = 'tv_proxy'
    description = 'TV export proxy'
    default_value = ''


class TvSnipDataBuild(SandboxTask):
    ad_hoc = []
    for b_type, b_id in Binaries:
        class_name = 'class_' + b_id
        class_descr = BinariesResources.description + ": " + b_id
        class_type = type(class_name, (BinariesResources,), {"name": b_id, "resource_type": b_type, "description": class_descr})
        ad_hoc.append(class_type)

    type = 'TV_SNIP_DATA_BUILD'
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = (ArcadiaTV, TvExportProxy) + tuple(ad_hoc)

    path = 'tv-build'

    def initCtx(self):
        self.ctx['kill_timeout'] = 48 * 60 * 60

    def checkout_arcadia(self, path):
        Arcadia.checkout(self.ctx['arcadia_tv'], path)

    def get_binaries(self, path):
        logging.info('searching for binaries')
        for b_type, b_id in Binaries:
            resource = None
            resource_id = self.ctx[b_id],
            logging.info('resource %s field %s id %s hz %s' % (b_type, b_id, resource_id,  self.ctx[b_id]))
            if self.ctx[b_id]:
                resource = channel.sandbox.get_resource(self.ctx[b_id])
            if not resource:
                resource = apihelpers.get_last_resource(resource_type=b_type)
            if not resource:
                raise SandboxTaskFailureError('Cannot find resource %s' % (b_type))
            resPath = self.sync_resource(resource)
            logging.info('copying binary %s to %s' % (resPath, path))
            copy_path(resPath, path)

    def build_tv(self, bin_path, arcadia_path, build_path):
        tv_proxy = ''
        if self.ctx['tv_proxy']:
            tv_proxy = 'http_proxy=' + self.ctx['tv_proxy']
        params = {"bin": bin_path, "arcadia": arcadia_path, "build": build_path, "proxy_var": tv_proxy}
        cmd = (
            'cd %(build)s && '
            'tar -zxf %(bin)s/class_date.lin_64.tgz && '
            'PERL5LIB=%(build)s/class_date && '
            '%(proxy_var)s SANDBOX=1 GMAKE=%(bin)s/gmake-3.81 %(arcadia)s/scripts/run.sh %(build)s || '
            'exit 1'
        ) % params
        process = None
        try:
            process = run_process(cmd, wait=True, shell=True, log_prefix='gmake_log')
        except Exception as e:
            logging.info('ut failed: %s' % e)
        if not process or process.returncode:
            raise SandboxTaskFailureError('Process %s died' % cmd)
        self.create_resource('Tv snip data regions', '%(build)s/common/regional_chains.tsv' % params, 'PROXY_WIZARD_TV_DATA_REGIONS')

    def arcadia_info(self):
        return None, "Tv snip data release", None

    def on_execute(self):
        if self.arch == 'any':
            raise SandboxTaskFailureError('You should select arch, "any" is not supported')

        make_folder(self.path)

        arcadia_path = self.abs_path('tv')
        bin_path = self.abs_path(os.path.join('tv', 'bin'))
        build_path = self.abs_path('build')

        for path in (arcadia_path, bin_path, build_path):
            logging.info('creating %s' % (path))
            os.makedirs(path)

        self.checkout_arcadia(arcadia_path)
        self.get_binaries(bin_path)

        self.build_tv(bin_path, arcadia_path, build_path)


__Task__ = TvSnipDataBuild
