# -*- coding: utf-8 -*-

import logging
import os
import threading

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.parameters import SandboxStringParameter, LastReleasedResource

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel

from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.projects import resource_types

from sandbox.projects.common.wizard.exception_manager import ExceptionManager

from sandbox.projects.common import apihelpers


Sources = resource_types.TV_SNIP_DATA_BUILD_GMAKE_SOURCES


class DirName(SandboxStringParameter):
    name = 'dir_name'
    description = 'name for gmake sources dir'
    default_value = 'make-3.81'


class BinName(SandboxStringParameter):
    name = 'bin_name'
    description = 'name for gmake binary'
    default_value = 'gmake-3.81'


class GmakeSources(LastReleasedResource):
    name = 'sources_resource_id'
    description = 'Sources tar.bz2'
    resource_type = Sources


class TvSnipDataBuildGmake(SandboxTask):
    type = 'TV_SNIP_DATA_BUILD_GMAKE'
    client_tags = ctc.Tag.GENERIC | ctc.Tag.LINUX_PRECISE

    input_parameters = (DirName, BinName, GmakeSources)

    @property
    def footer(self):
        return '<pre>{}</pre>'.format(self.ctx.get('process_log'))

    def initCtx(self):
        self.ctx['process_log'] = ''

    def get_sources(self, path):
        logging.info('searching for sources')
        for type in [Sources]:
            resource = None
            if self.ctx['sources_resource_id']:
                resource = channel.sandbox.get_resource(self.ctx['sources_resource_id'])
            if not resource:
                resource = apihelpers.get_last_resource(resource_type=type)
            if not resource:
                raise SandboxTaskFailureError('Cannot find resource %s' % (type))

            resPath = self.sync_resource(resource)
            logging.info('copying sources %s to %s' % (resPath, path))
            copy_path(resPath, path)

    def logged_process(self, cmd, shell=False):
        log_filename = self.abs_path('log.txt')
        log = open(log_filename, 'w')
        process = None
        try:
            process = run_process(cmd, stdout=log, stderr=log, wait=True, shell=shell)
        except Exception as e:
            logging.info('ut failed: %s' % e)
        log.close()
        self.ctx['process_log'] = open(log_filename).read()
        if not process or process.returncode:
            raise SandboxTaskFailureError('Process %s died' % cmd)

    def build_gmake(self, build_path, out_path):
        if not self.ctx.get('dir_name'):
            self.ctx['dir_name'] = 'make-3.81'
        if not self.ctx.get('bin_name'):
            self.ctx['bin_name'] = 'gmake'

        params = {
            "build": build_path,
            "src": os.path.join(build_path, self.ctx['dir_name']),
            "out": os.path.join(out_path, self.ctx['bin_name']),
        }

        self.logged_process(
            'cd %(build)s\
                 && tar -jxf `ls *make*t*bz* | head -n 1`\
                 && cd %(src)s\
                 && ./configure --enable-static\
                 && ./build.sh\
                 && cp make %(out)s' % params,
            shell=True,
        )

        self.create_resource('Tv snip data gmake', params["out"], 'TV_SNIP_DATA_BUILD_GMAKE')

    def arcadia_info(self):
        return None, "Tv snip data gmake release", None

    def on_execute(self):
        if self.arch == 'any':
            raise SandboxTaskFailureError('You should select arch, "any" is not supported')

        make_folder(self.path)
        build_path = self.abs_path('build')
        out_path = self.abs_path('out')

        os.makedirs(build_path)
        os.makedirs(out_path)

        emanager = ExceptionManager()

        threads = (
            threading.Thread(target=ExceptionManager.run_target, args=(emanager, TvSnipDataBuildGmake.get_sources, self, build_path)),
        )

        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()

        emanager.check_exceptions()

        self.build_gmake(build_path, out_path)


__Task__ = TvSnipDataBuildGmake
