# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.TvSnipDataBuild import TvSnipDataBuild


class ReleaseDescription(SandboxStringParameter):
    name = 'release_description'
    description = 'Release description'
    multiline = True
    default_value = 'New tv snip data build release'


class TvSnipDataRelease(SandboxTask):
    type = 'TV_SNIP_DATA_RELEASE'
    client_tags = ~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE

    input_parameters = tuple([ReleaseDescription]) + TvSnipDataBuild.input_parameters

    def initCtx(self):
        self.ctx['kill_timeout'] = 7 * 60 * 60

    def on_execute(self):
        if self.arch == 'any':
            raise SandboxTaskFailureError('You should select arch, "any" is not supported')

        params = {
            'kill_timeout': max(7*60*60, int(self.ctx['kill_timeout'])),
        }
        for p in TvSnipDataBuild.input_parameters:
            params[p.name] = self.ctx[p.name]

        if not self.ctx.get('build_task'):
            logging.info('Create build quentao task.')
            build_task = self.create_subtask(
                task_type='TV_SNIP_DATA_BUILD',
                arch=self.arch,
                execution_space=self.execution_space,
                input_parameters=params,
                description='build data for tv snip'
            )
            self.ctx['build_task'] = build_task.id
            self.wait_task_completed(build_task)
        else:
            build_task = channel.sandbox.get_task(self.ctx['build_task'])
            if build_task.is_failure():
                raise SandboxTaskFailureError('TvSnipDataBuild task {task_id} was failed.'.format(
                    task_id=build_task.id
                ))
            if build_task.is_finished():
                release = self.create_release(
                    build_task.id,
                    subject='tv_snip_data_build {task_id}'.format(task_id=build_task.id),
                    comments=self.ctx.get('release_description'),
                    addresses_to='hob@yandex-team.ru',
                )
                if not release:
                    raise SandboxTaskFailureError('Cannot release task {task_id}.'.format(task_id=build_task.id))
            else:
                raise SandboxTaskFailureError('TvSnipDataBuild task {task_id} was not completed correctly.'.format(task_id=build_task.id))


__Task__ = TvSnipDataRelease
