import requests
import simplejson
import yt.wrapper
from datetime import datetime


def abuse_join_premap(record):
    is_abuse = record['@table_index']
    record['is_abuse'] = is_abuse
    del record['@table_index']

    if is_abuse:
        record['sort_timestamp'] = record['RemoderationRequestTime']
        record['bid'] = record['Bid']
        del record['Bid']
    else:
        timestamp = datetime.strptime(record['timeChanged'], '%Y-%m-%d %H:%M:%S')
        record['sort_timestamp'] = int(timestamp.strftime('%s'))

    yield record


def abuse_join_reduce(keys, records):

    banner = None
    for record in records:

        if record['is_abuse']:
            abuse = record
            abuse.update(banner or {})
            del record['sort_timestamp']
            del record['is_abuse']
            yield abuse
        else:
            banner = record


def update_remoderation_statuses(abuse_records, check_remoderation_status_url):

    bids = set(int(abuse_record['Bid']) for abuse_record in abuse_records)

    call_object = simplejson.dumps({
        "method": "checkRemoderateStatusAbuseBanners",
        "params": [[{'bid': bid} for bid in bids]],
        "jsonrpc": "2.0",
    })

    response = requests.post(
        check_remoderation_status_url,
        headers={'content-type': 'application/json'},
        data=call_object,
        timeout=10
    )
    response.raise_for_status()

    statuses = {}
    for record in response.json()['result']:
        bid = int(record['bid'])
        if 'error' in record:
            statuses[bid] = None
        else:
            status = record.get('status')
            statuses[bid] = None if status == 'None' else status

    for abuse_record in abuse_records:
        abuse_record['RemoderationStatus'] = statuses[int(abuse_record['Bid'])]
        yield abuse_record


def update_abuse_remoderation_requests_table(abuse_remoderation_requests_table, yt_client, check_remoderation_status_url):
    abuse_records_with_status = []
    abuse_records_without_status = []

    for abuse_record in list(yt_client.read_table(abuse_remoderation_requests_table, raw=False)):
        if abuse_record.get('remoderation_status'):
            abuse_records_with_status.append(abuse_record)
        else:
            abuse_records_without_status.append(abuse_record)

    batch_size = 500
    for start_index in xrange(0, len(abuse_records_without_status), batch_size):
        abuse_records_batch = abuse_records_without_status[start_index:start_index + batch_size]
        abuse_records_batch = update_remoderation_statuses(abuse_records_batch, check_remoderation_status_url)
        abuse_records_with_status.extend(abuse_records_batch)

    yt_client.write_table(abuse_remoderation_requests_table, abuse_records_with_status, raw=False)


def join_abuses(abuse_remoderation_requests_table, prod_banners_table, dst_table, yt_client):

    yt_client.run_map_reduce(
        abuse_join_premap,
        abuse_join_reduce,
        [
            yt.wrapper.TablePath(prod_banners_table, start_index=3900000000, client=yt_client),
            yt.wrapper.TablePath(abuse_remoderation_requests_table, client=yt_client),
        ],
        dst_table,
        sort_by=['bid', 'sort_timestamp', 'is_abuse'],
        reduce_by=['bid'],
        format=yt.YsonFormat(control_attributes_mode="row_fields"),
    )


def run_update(yt_client, dst_table, prod_banners_table, abuse_remoderation_requests_table, check_remoderation_status_url):
    update_abuse_remoderation_requests_table(
        abuse_remoderation_requests_table,
        yt_client=yt_client,
        check_remoderation_status_url=check_remoderation_status_url
    )
    join_abuses(
        abuse_remoderation_requests_table=abuse_remoderation_requests_table,
        prod_banners_table=prod_banners_table,
        dst_table=dst_table,
        yt_client=yt_client
    )
