# -*- coding: utf-8 -*-
import os
import time

from sandbox import sdk2
from sandbox.projects import resource_types

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.paths import make_folder
from sandbox.projects.common.geosearch.utils import pack_dir


class UpdateBusinessRatings(sdk2.Task):
    """
        Update business ratings, get info from YT via YQL and generate ratings
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql')
        ]

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        self.mark_released_resources('stable')

    def on_execute(self):
        ratings_dir = make_folder('ratings')
        self.update_ratings(ratings_dir)
        new_path = 'ratings.tar.gz'
        pack_dir(ratings_dir, new_path, arch_type='tar.gz')
        sdk2.ResourceData(resource_types.BUSINESS_RATINGS(
            self, "update business ratings for offline search business indexer", new_path
        ))

    def update_ratings(self, rating_dir):
        timestamp_path = os.path.join(rating_dir, 'timestamp')
        ratings_xml_path = os.path.join(rating_dir, 'ratings.xml')

        with open(timestamp_path, 'w') as fout:
            fout.write(time.ctime())

        rating_rows = self.load_ratings_from_yt()

        with open(ratings_xml_path, 'w') as fout:
            fout.write('<BusinessRatings>\n')
            for id, score, ratings in rating_rows:
                fout.write(
                    '\t<BusinessRating id="{id}">'
                    '<score>{score}</score>'
                    '<ratings>{ratings}</ratings>'
                    '<reviews>0</reviews>'
                    '</BusinessRating>\n'
                    .format(**locals())
                )
            fout.write('</BusinessRatings>')

    def load_ratings_from_yt(self):
        from yql.api.v1.client import YqlClient
        client = YqlClient(token=sdk2.Vault.data(self.owner, 'YQL_TOKEN'))
        request = client.query("""
            SELECT
                id,
                Rating.Rating,
                Rating.Amount
            FROM (
                SELECT export_proto.Id AS id, export_proto.GeosearchData.Rating AS Rating
                FROM hahn.`//home/sprav/altay/prod/geosearch_snapshot/company`
            )
            FLATTEN LIST BY Rating
            WHERE
                Rating.Aref == "yandex"
        """)
        request.run()

        for table in request.get_results():
            table.fetch_full_data()
            return table.rows
