# coding=utf8

import os
import cgi
import tempfile
import re
import logging

from sandbox.sandboxsdk.parameters import SandboxSelectParameter, SandboxStringParameter, SandboxIntegerParameter, SandboxBoolParameter
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.projects import resource_types
from sandbox.sdk2 import yav

# import sandbox.common.types.misc as ctm

from sandbox.projects.common.gencfg import utils as config_generator_utils
from sandbox.projects.common.gencfg.task import IGencfgBuildTask
import sandbox.common.types.task as ctt
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.gencfg.MakeGencfgDnsResource as gencfg_dns_task


class EUpdaterStatus:
    SUCCESS = 'SUCCESS'
    NOTHINGTOCOMMIT = 'NOTHINGTOCOMMIT'
    FAILURE = 'FAILURE'


class IUpdater(object):
    def __init__(self, descr):
        self.descr = descr

    def run_process_and_get_log(self, args, task, log_prefix, external_env={}):
        if external_env:
            custom_env = external_env
        else:
            custom_env = os.environ.copy()
            custom_env['GENCFG_DEFAULT_OAUTH'] = task.get_vault_data('GENCFG', 'gencfg_oauth')
            custom_env['YT_TOKEN'] = task.get_vault_data('GENCFG', 'yt-token-robot-gencfg2')

        p = run_process(args, work_dir=task.get_gencfg_path(), environment=custom_env, log_prefix=log_prefix, outputs_to_one_file=False)
        if p.returncode != 0:
            raise Exception("Process <%s> failed with exit status %s" % (args, p.returncode))
        return open(os.path.join(get_logs_folder(), '%s.out.txt' % log_prefix)).read().strip()

    def find_log_entry(self, log, reg, strip_prefix=False):
        filtered = filter(lambda x: re.match(reg, x) is not None, log)

        if len(filtered) > 1:
            print("More that 1 line with prefix <%s> in log" % reg)
            return None
        if len(filtered) == 0:
            print("Not found line with prefix <%s> in log" % reg)
            return None

        if strip_prefix:
            full_reg = "^%s(.*)" % reg
            return re.match(full_reg, filtered[0]).group(1)
        else:
            return filtered[0].strip()

    def can_modify(self):
        """If updater modifies gencfg db or not"""
        return True

    def process(self, dbpath):
        raise NotImplementedError("Unimplemened method process in updater %s" % self.id)


"""
    Update invnums of all hosts with local bot queries
"""


class TInvnumUpdater(IUpdater):
    def __init__(self, descr):
        super(TInvnumUpdater, self).__init__(descr)

    def process(self, task):
        run_process(['utils/pregen/rename_hosts.py', '-m', 'oops', '-s', 'ALL', '--filter', 'lambda x: x.is_vm_guest() == False', '-i'],
                    work_dir=task.get_gencfg_path(), log_prefix='update_hosts')

        return (EUpdaterStatus.SUCCESS, map(lambda x: x.strip(), open(os.path.join(get_logs_folder(), 'update_hosts.out.txt')).readlines()))


"""
    Update location of all hosts with local bot queries
"""


class THostLocationUpdater(IUpdater):
    def __init__(self, descr):
        super(THostLocationUpdater, self).__init__(descr)

    def process(self, task):
        custom_env = os.environ.copy()
        custom_env['GENCFG_DEFAULT_OAUTH'] = task.get_vault_data('GENCFG', 'gencfg_default_oauth')
        custom_env['YT_TOKEN'] = task.get_vault_data('GENCFG', 'yt-token')

        run_process(['./utils/pregen/update_hosts.py', '-a', 'updateoops', '-s', 'ALL', '--host-filter', 'lambda x: x.is_vm_guest() == False', '--ignore-detect-fail', '-y', '-v'],
                    work_dir=task.get_gencfg_path(), log_prefix='update_hosts', environment=custom_env)
        run_process(['./utils/pregen/update_hosts.py', '-a', 'updateoops', '-i', 'walle_tags,model', '--update-to-def-value', '-s', 'ALL', '-y', '-v'],
                    work_dir=task.get_gencfg_path(), log_prefix='update_walle_tags', environment=custom_env)

        return (EUpdaterStatus.SUCCESS, [
            open(os.path.join(get_logs_folder(), 'update_hosts.out.txt')).readline().strip(),
        ])


"""
    Update ipv6/ipv4 addrs of all hosts with local bot queries
"""


class TIpaddrsUpdater(IUpdater):
    def __init__(self, descr):
        super(TIpaddrsUpdater, self).__init__(descr)

    def process(self, task):
        changelog = []
        for proto in ['ipv4addr', 'ipv6addr']:
            log_prefix = 'update_hosts_%s' % proto
#            run_process(['utils/pregen/update_hosts.py', '-a', 'updatelocal', '-i', proto, '-s', 'sas1-3364.search.yandex.net', '-y', '--update-to-def-value'],
            run_process(['utils/pregen/update_hosts.py', '-a', 'updatelocal', '-i', proto, '-s', 'ALL_RUNTIME,ALL_SEARCH,ALL_YT', '--host-filter', 'lambda x: x.is_vm_guest() == False', '-y', '-v'],
                        work_dir=task.get_gencfg_path(), log_prefix=log_prefix)

            changelog.append("Updating proto %s: %s" % (proto, open(os.path.join(get_logs_folder(), '%s.out.txt' % log_prefix)).readline().strip()))

        return (EUpdaterStatus.SUCCESS, changelog)


"""
    Update raw bot info of memory,disk,ssd
"""


class TBotHwUpdater(IUpdater):
    def __init__(self, descr):
        super(TBotHwUpdater, self).__init__(descr)

    def process(self, task):
        changelog = []
        log_prefix = 'update_hosts_botmem_botdisk_botssd'
        update_fields = 'botmem,botdisk,botssd,golemowners,gpu_count,gpu_models'
        run_process(['utils/pregen/update_hosts.py', '-a', 'updatelocal', '-i', update_fields, '-s', 'ALL', '--host-filter', 'lambda x: x.is_vm_guest() == False', '-y'],
                        work_dir=task.get_gencfg_path(), log_prefix=log_prefix)
        changelog.append("Updating botmem,botdisk,botssd,golemowners: %s" % (open(os.path.join(get_logs_folder(), '%s.out.txt' % log_prefix)).readline().strip()))

        return (EUpdaterStatus.SUCCESS, changelog)


"""
    Peform all sync with bot actions:
     - add extra bot hosts (move working to reserved, unworking to ALL_PRENALIVKA)
     - move working hosts from ALL_PRENALIVKA to RESERVED
     - remove rusty hosts which are not in bot dump
"""


class TSyncWithBotUpdater(IUpdater):
    def __init__(self, descr):
        super(TSyncWithBotUpdater, self).__init__(descr)

    def process(self, task):
        changelog = []

        sublog = self.run_process_and_get_log(['recluster2/prepare.sh', 'rename_by_invnum'], task, 'rename_by_invnum')

        renames = map(lambda x: x.replace('Need to rename: ', '    '), sublog.split('\n'))
        changelog.append('Renames:\n%s' % ('\n'.join(renames)))

        sublog = self.run_process_and_get_log(['recluster2/prepare.sh', 'sync_with_bot'], task, 'sync_with_bot').split('\n')

        for pattern in (
            "Added \d+ hosts to ALL_PRENALIVKA:", "Added \d+ hosts to ALL_SANDBOX:", "Removed \d+ hosts from ALL_PRENALIVKA:",
            "Move \d+ hosts to reserved:", "Removed \d+ hosts from ALL_UNSORTED:"
        ):
            founded = self.find_log_entry(sublog, pattern)
            if founded is not None:
                changelog.append(founded)

        return (EUpdaterStatus.SUCCESS, changelog)


class TBusyHostsUpdater(IUpdater):
    def __init__(self, descr, busy_group, move_group=None, mail_admins=True):
        super(TBusyHostsUpdater, self).__init__(descr)
        self.busy_group = busy_group
        self.move_group = move_group
        self.mail_admins = mail_admins

    def process(self, task):
        args = ['./utils/sky/show_hosts_with_raised_instances.py', '-a', 'show_all', '-s', self.busy_group, '-p', 'instancestate']
        if self.move_group is None:
            args.extend(['-r'])
        else:
            args.extend(['--move-to-group', self.move_group])

        sublog = self.run_process_and_get_log(args, task, 'move_unraised_instances').split('\n')

        changelog = []

        move_to_reserved = self.find_log_entry(sublog, "Unused hosts:", True)
        if self.move_group is None:
            changelog.append("Unused hosts (move to reserved): %s" % move_to_reserved)
        else:
            changelog.append("Unused hosts (move to group %s): %s" % (self.move_group, move_to_reserved))
        changelog.append("Failure hosts: %s" % self.find_log_entry(sublog, "Failure hosts:", True))
        changelog.append("Busy hosts: %s" % self.find_log_entry(sublog, "Used hosts:", True))

        if self.mail_admins:
            subject = "Машины, на которых нужно опустить инстансы"
            prefix = """Добрый день.

Машины указанные ниже, предназначены к переносу в резерв/демонтаж, однако на них до сих пор есть неопущенные инстансы.
Просьба администраторов опустить инстансы своих проектов.

"""
            body = '\n'.join(sublog[sublog.index("Still used hosts:"):])

            fhandle, fname = tempfile.mkstemp()
            os.write(fhandle, prefix + body)
            os.close(fhandle)

            try:
                watchers = [task.author]
                if task.ctx['watchers'] != '':
                    watchers.extend(map(lambda x: x.strip(), task.ctx['watchers'].split(',')))

                mail_args = ['./utils/common/spam_admins.py', '-s', subject, '-c', '#%s' % fname, '-g', '-d', '-v', '-r', ','.join(watchers)]
                run_process(mail_args, work_dir=task.get_gencfg_path(), log_prefix='mail_admins')
            finally:
                os.remove(fname)

        return (EUpdaterStatus.SUCCESS, changelog)


"""
    Find all groups with set properties.expires. Delete expired groups, warn on groups, which will be expired soon
"""


class TTemporaryGroupsUpdater(IUpdater):
    WARN_DAYS = 8

    class EStatus:
        DELETE = 'DELETE'
        WARN = 'WARN'

    def __init__(self, descr):
        self.descr = descr

    def parse_log(self, log, status):
        result = []
        for line in log.split('\n'):
            line = line.strip()
            if line == '':
                continue

            groupname, owners, watchers, expiration_date = line.split('\t')
            owners = owners.split(',') if owners != '' else []
            watchers = watchers.split(',') if watchers != '' else []

            result.append((groupname, owners + watchers, expiration_date, status))

        return result

    def process_expired_group(self, task, group_data, changelog):
        groupname, owners, expiration_date, status = group_data

        if status == TTemporaryGroupsUpdater.EStatus.WARN:
            mail_subj = "[sandbox] [update_config_generator_db] [%s] Временная группа <%s> будет автоматически удалена <%s>" % (task.id, groupname, expiration_date)
            mail_body = (
                "Добрый день.\n\n"
                "Указанная в названии временная группа вскоре будет удалена. "
                "Если вы не считаете, что она удаляется преждевременно, сообщите об этом админам gencfg, "
                "либо поправьте время удаления группы самостоятельно в web-интерфейсе, "
                "поле Expiration Date (если вы владелец группы): https://gencfg.yandex-team.ru/trunk/groups/%s#properties"
            ) % groupname
            channel.sandbox.send_email(owners, '', mail_subj, mail_body)
            changelog.append("Mailed to <%s> : temporary group <%s> is about to expire at <%s>" % (",".join(owners), groupname, expiration_date))
        else:
            try:
                run_process(
                    ["./utils/common/update_igroups.py", "-a", "remove", "-g", groupname, "-r", "-c", "ALL_UNWORKING_BUSY"],
                    work_dir=task.get_gencfg_path(), log_prefix="remove_group_%s" % groupname
                )
                changelog.append("Removed group <%s>" % groupname)

                try:
                    mail_subj = "[sandbox] [update_config_generator_db] [%s] Временная группа <%s> автоматически удалена изза истекшего срока использования" % (task.id, groupname)
                    mail_body = (
                        "Добрый день.\n\n"
                        "Указанная в названии временная группа была удалена из-за того, что у нее вышел срок использования"
                    )

                    channel.sandbox.send_email(owners, '', mail_subj, mail_body)
                    changelog.append("Mailed to <%s> : temporary group <%s> deleted due to expiration" % (",".join(owners), groupname))
                except:
                    pass

            except:
                changelog.append("Failed to remove group <%s>" % groupname)

    def process(self, task):
        entries = []

        args = ['./utils/common/show_groups.py', '-E', '-i', 'name,owners,watchers,expiration_date']
        sublog = self.run_process_and_get_log(args, task, 'get_expired')
        entries.extend(self.parse_log(sublog, TTemporaryGroupsUpdater.EStatus.DELETE))

        args = [
            './utils/common/show_groups.py',
            '-i', 'name,owners,watchers,expiration_date',
            '-f', (
                'lambda x: '
                'x.card.master is None and '
                'x.card.properties.expires is not None and '
                'x.card.properties.expires.date - datetime.date.today() < datetime.timedelta(%s) and '
                'x.card.properties.expires.date - datetime.date.today() >= datetime.timedelta(0)'
            ) % TTemporaryGroupsUpdater.WARN_DAYS
        ]
        sublog = self.run_process_and_get_log(args, task, 'get_almost_expired')
        entries.extend(self.parse_log(sublog, TTemporaryGroupsUpdater.EStatus.WARN))

        changelog = []
        for entry in entries:
            self.process_expired_group(task, entry, changelog)

        return (EUpdaterStatus.SUCCESS, changelog)


class THbUpdater(IUpdater):
    def __init__(self, descr):
        super(THbUpdater, self).__init__(descr)

    def process(self, task):
        args = ['./utils/pregen/update_hosts.py', '-s', 'ALL', '-i', 'disk,ssd,memory', '-a', 'updatehb', '--ignore-detect-fail', '--update-to-def-value', '-y']
        sublog = self.run_process_and_get_log(args, task, 'update_from_hb').split('\n')

        changelog = []

        # Get overall statistics
        changelog.append(self.find_log_entry(sublog, "Overall statistics:"))

        # find hosts which can not be updated due to changes in memory/cpu model/disk size
        can_not_update_entries = map(lambda x: x.split(' ')[4], filter(lambda x: x.startswith("!!! Can not update"), sublog))
        changelog.append("Can not update due to hw changes (%d machines): %s" % (len(can_not_update_entries), ",".join(can_not_update_entries)))

        return (EUpdaterStatus.SUCCESS, changelog)


class TNannyDynamicUpdater(IUpdater):
    def __init__(self, descr):
        super(TNannyDynamicUpdater, self).__init__(descr)

    def process(self, task):
        args = ['./utils/common/import_nanny_dynamic.py', '-g', 'ALL_NANNY_DYNAMIC', '-v', '-v', '-y']
        sublog = self.run_process_and_get_log(args, task, 'import_nanny_dynamic').split('\n')

        changelog = []
        changelog.append(self.find_log_entry(sublog, "Total \d+ nanny groups, to add \d+, to remove \d+, to modify \d+"))
        changelog.append(self.find_log_entry(sublog, "    Added to gencfg:"))
        changelog.append(self.find_log_entry(sublog, "    Removed from gencfg:"))
        changelog.append(self.find_log_entry(sublog, "    Modified in gencfg:"))

        return (EUpdaterStatus.SUCCESS, changelog)


class TUworkingCycler(IUpdater):
    def __init__(self, descr):
        super(TUworkingCycler, self).__init__(descr)

    def process(self, task):
        args = ["./utils/pregen/check_alive_machines.py", "-c", "sshport", "-g", "ALL_UNWORKING", "--add-working-to-group", "ALL_UNWORKING_BUSY"]
        sublog = self.run_process_and_get_log(args, task, 'cycle_unworking').split('\n')

        changelog = []
        changelog.append(self.find_log_entry(sublog, "Working hosts \(\d+ total\):"))

        return (EUpdaterStatus.SUCCESS, changelog)


class TKpiUpdater(IUpdater):
    def __init__(self, descr):
        super(TKpiUpdater, self).__init__(descr)

    def process(self, task):
        args = ["./utils/common/update_kpi_graphs.py", "-a", "update"]
        self.run_process_and_get_log(args, task, "update_kpi_db")

        args = ['./utils/common/manipulate_statface.py', '-a', 'updatedata']
        self.run_process_and_get_log(args, task, 'update_statface_kpi')

        return (EUpdaterStatus.SUCCESS, [])


class TFiredOwnersUpdater(IUpdater):
    """Remove all fired owners from group owners/watchers"""

    def __init__(self, descr):
        super(TFiredOwnersUpdater, self).__init__(descr)

    def process(self, task):
        args = ['./utils/common/change_groups_owners.py', '-a', 'fixfired', '-g', 'ALL', '-y']
        self.run_process_and_get_log(args, task, "remove_fired_group_owners")

        return (EUpdaterStatus.SUCCESS, [])


class TPsiReservedUpdater(IUpdater):
    """Updated list of hosts in PSI groups, connected to {MSK,SAS,MAN,VLA}_RESERVED"""

    def __init__(self, descr):
        super(TPsiReservedUpdater, self).__init__(descr)

    def process(self, task):
        for group_name in ('MAN_PSI_RESERVED_PORTOVM', 'SAS_PSI_RESERVED_PORTOVM', 'VLA_PSI_RESERVED_PORTOVM', 'MSK_PSI_RESERVED_PORTOVM',):
            args = ['./tools/recluster/main.py', '-a', 'recluster', '-g', group_name, '-c', 'all']
            self.run_process_and_get_log(args, task, 'recluster_psi_reserved_for_{}'.format(group_name.lower()))

        return (EUpdaterStatus.SUCCESS, [])


class TDnsCacheUpdater(IUpdater):
    """Update dns cache of virtual machines"""

    def __init__(self, descr):
        super(TDnsCacheUpdater, self).__init__(descr)

    def process(self, task):
        args = ['./utils/common/manipulate_dnscache.py', '-a', 'update']
        self.run_process_and_get_log(args, task, 'update_dnscache')

        return (EUpdaterStatus.SUCCESS, [])


class TSlbCacheUpdater(IUpdater):
    """Update slb cache from racktables (GENCFG-1781)"""

    def __init__(self, descr):
        super(TSlbCacheUpdater, self).__init__(descr)

    def process(self, task):
        args = ['./utils/common/manipulate_slbs.py', '-a', 'sync']
        self.run_process_and_get_log(args, task, 'update_slbcache')

        return (EUpdaterStatus.SUCCESS, [])


class TMongoHbfDnsUpdater(IUpdater):
    """Update table <gencfg_dns> in mongo DB <topology_commits> with new gencfg instances"""

    def __init__(self, descr):
        super(TMongoHbfDnsUpdater, self).__init__(descr)

    def can_modify(self):
        return False

    def process(self, task):
        args = ['./utils/mongo/populate_dns.py', '-a', 'export']
        self.run_process_and_get_log(args, task, "populate_dns")

        make_dns_bundle_task = gencfg_dns_task.MakeGencfgDnsResource(
            None, owner=task.owner, description="child task of {}".format(self)
        )
        make_dns_bundle_task.enqueue()

        print('spawned child MakeGencfgDnsResource task {}'.format(make_dns_bundle_task.id))
        task.set_info(
            'spawned child MakeGencfgDnsResource task: <a href="https://sandbox.yandex-team.ru/task/{id}/view">{id}</a>'.format(id=make_dns_bundle_task.id),
            do_escape=False,
        )
        return (EUpdaterStatus.SUCCESS, [])


class TNannyServicesCacheUpdater(IUpdater):
    """Update nanny services cache (RX-354)"""

    def __init__(self, descr):
        super(TNannyServicesCacheUpdater, self).__init__(descr)

    def can_modify(self):
        return False

    def process(self, task):
        args = ['./utils/common/manipulate_nanny_services.py', '-a', 'download', '-o', task.get_nanny_services_file()]
        custom_env = os.environ.copy()
        custom_env['GENCFG_DEFAULT_OAUTH'] = task.get_vault_data('GENCFG', 'gencfg_nanny_oauth')
        custom_env['YT_TOKEN'] = task.get_vault_data('GENCFG', 'yt-token-robot-gencfg2')

        self.run_process_and_get_log(args, task, 'update_nanny_services_cache', custom_env)

        task._create_resource('Cached nanny services for gencfg', task.get_nanny_services_file(), resource_types.GENCFG_NANNY_SERVICES, arch='any')

        return (EUpdaterStatus.SUCCESS, [])


class TSyncWithStaff(IUpdater):
    """Sync staff groups (RX-474)"""

    def __init__(self, descr):
        super(TSyncWithStaff, self).__init__(descr)

    def process(self, task):
        # sync staff
        args = ['./utils/common/manipulate_staff.py', '-a', 'sync']
        self.run_process_and_get_log(args, task, 'sync_with_staff')

        return (EUpdaterStatus.SUCCESS, [])


class TSyncWithAbc(IUpdater):
    """Sync staff groups (RX-447)"""

    def __init__(self, descr):
        super(TSyncWithAbc, self).__init__(descr)

    def process(self, task):
        # sync staff
        args = ['./utils/common/manipulate_abc.py', '-a', 'sync']
        self.run_process_and_get_log(args, task, 'sync_with_abc')

        return (EUpdaterStatus.SUCCESS, [])


class TSyncWithNanny(IUpdater):
    """Sync group nanny services (RX-532)"""

    def __init__(self, descr):
        super(TSyncWithNanny, self).__init__(descr)

    def process(self, task):
        # sync staff
        args = ['./utils/common/manipulate_nanny_services.py', '-a', 'sync_groups']
        self.run_process_and_get_log(args, task, 'sync_with_nanny')

        return (EUpdaterStatus.SUCCESS, [])


class TIpv4TunnelsUpdater(IUpdater):
    """Sync group nanny services (RX-526)"""

    def __init__(self, descr):
        super(TIpv4TunnelsUpdater, self).__init__(descr)

    def process(self, task):
        # remove old from local database
        args = ['./utils/common/manipulate_ipv4tunnels.py', '-a', 'cleanup']
        self.run_process_and_get_log(args, task, 'cleanup_local_base')

        # upload data to racktables
        secret = yav.Secret("sec-01f7rjxh61ab4sk01qe2kehdmf", "ver-01f7rjxh6gt4h1gd4t1y2006t9")

        args = ['./utils/common/manipulate_ipv4tunnels.py', '-a', 'export'
                , '--token', secret.data()["racktables-token"]]
        self.run_process_and_get_log(args, task, 'export_to_racktables')

        return (EUpdaterStatus.SUCCESS, [])


UPDATERS = {
    "invnum": TInvnumUpdater("Rename hosts based on invnum"),
    "ipaddrs": TIpaddrsUpdater("Update ip addresses of all hosts"),
    "hwlocal": THostLocationUpdater("Update invnum/dc/queue/switch/rack/vlan for all hosts"),
    "syncwithbot": TSyncWithBotUpdater("Sync gencfg db with bot"),
    "cleanbusy": TBusyHostsUpdater("Move free hosts to reserved group", "ALL_UNWORKING_BUSY", mail_admins=True),
    "cleantemporary": TTemporaryGroupsUpdater("Clean expired temporary groups"),
    "updatehb": THbUpdater("Update hosts hwdata from heartbeat databse"),
    "hwbot": TBotHwUpdater("Update botmem/botdisk/botssd"),
    "importnannydynamic": TNannyDynamicUpdater("Import nanny dynamic into gencfg"),
    "cycleunworking": TUworkingCycler("Cycle unworking machines (move working machines from ALL_UNWORKING group)"),
    "updatekpi": TKpiUpdater("Update kpi data/graphics"),
    "removefiredowners": TFiredOwnersUpdater("Remove fired group owners/watchers"),
    "updatepsireserved": TPsiReservedUpdater("Update PSI groups on reserved"),
    "updatednscache": TDnsCacheUpdater("Update dns cache for vm machines"),
    "updateslbcache": TSlbCacheUpdater("Update slb cache"),
    "populatedns": TMongoHbfDnsUpdater("Export hbf instances dns cache to mongo"),
    "updatenannyservicescache": TNannyServicesCacheUpdater("Updated nany services cache"),
    "syncwithstaff": TSyncWithStaff("Sync staff groups"),
    "syncwithabc": TSyncWithAbc("Sync abc groups"),
    "syncwithnanny": TSyncWithNanny("Sync nanny services info"),
    "updateipv4tunnels": TIpv4TunnelsUpdater("Update ipv4tunnels"),
}


class GeneratorUpdateEntityParameter(SandboxSelectParameter):
    name = "update_entity"
    description = "Which entity of db update"
    default_value = None
    required = True
    choices = map(lambda (x, y): (y.descr, x), UPDATERS.iteritems())


class ExtraWatchersParameter(SandboxStringParameter):
    name = 'watchers'
    description = 'Extra users to send mail and other stuff (comma-separated)'
    default_value = ''


class MaxDiffSize(SandboxIntegerParameter):
    name = 'max_diff_size'
    description = 'If generate diff flag is true the total diff size muse be limited '
    default_value = 3003974


class IgnoreDiffLimit(SandboxBoolParameter):
    name = 'ignore_diff_limit'
    description = 'Ignore max diff size limit '
    default_value = False


class UpdateConfigGeneratorDb(nanny.ReleaseToNannyTask2, IGencfgBuildTask):
    type = "UPDATE_CONFIG_GENERATOR_DB"

    input_parameters = [GeneratorUpdateEntityParameter, ExtraWatchersParameter, MaxDiffSize, IgnoreDiffLimit]
    required_ram = 3 * 1024

    def on_enqueue(self):
        if self.ctx['update_entity'] in {'hwlocal', 'syncwithbot', 'ipaddrs', 'invnum', 'updatehb', 'hwbot', 'syncwithabc'}:
            self.semaphores(ctt.Semaphores(
                acquires=[ctt.Semaphores.Acquire(name='GENCFG_SDK_SYNC_SCHEDULER_HOSTDATA_SEMAPHORE', capacity=1)],
            ))
        else:
            logging.info('SKIP acquires semaphore becasuse update_entity == {}'.format(self.ctx['update_entity']))

    @property
    def footer(self):
        return '<br/>'.join([
            '<h4>STATUS {}</h4>'.format(self.ctx.get('status')),
            '<h4>Changelog </h4>',
            '<br/>'.join(self.ctx.get('escaped_changelog', self.ctx.get('changelog', ['No changelog yet']))),
            '<br/><h4>Diffbuilder result </h4>',
            '<pre>{}</pre>'.format(self.ctx.get('diffbuilder_result', 'No diffbuilder result'))
        ])

    def get_nanny_services_file(self):
        return self.abs_path('nanny.services')

    def on_execute(self):
        config_generator_utils.clone_gencfg_all(self.get_gencfg_path(), 'full')
        run_process(["bash", "./install.sh"], work_dir=self.get_gencfg_path(), log_prefix="install")  # prepare venv

        try:
            updater = UPDATERS[self.ctx['update_entity']]
            self.ctx['status'], self.ctx['changelog'] = updater.process(self)
            self.ctx['escaped_changelog'] = map(lambda x: cgi.escape(x), self.ctx['changelog'])

            if self.ctx['status'] == EUpdaterStatus.SUCCESS and updater.can_modify():
                # check if build ok
                p = run_process(["bash", "./gen.sh", "run_checks"], work_dir=self.get_gencfg_path(), log_prefix="check_after_update")
                if p.returncode != 0:
                    raise Exception('Process ./gen.sh failed with code <{}>'.format(p.retruncode))

                # upload hosts data
                custom_env = os.environ.copy()
                custom_env['GENCFG_DEFAULT_OAUTH'] = self.get_vault_data('GENCFG', 'gencfg_default_oauth')
                hosts_data_commit_msg = '[sandbox] [update_{}] Autoupdated by <{}> in sandbox task {}'.format(self.ctx['update_entity'], updater.descr, self.http_url())
                run_process(['./utils/common/manipulate_hostsdata.py', '-a', 'upload', '-m', hosts_data_commit_msg, '--external-workdir', os.getcwd(), '--switch-to'],
                            work_dir=self.get_gencfg_path(), environment=custom_env, log_prefix='upload_hosts_data')

                run_process(["svn", "diff"], work_dir=os.path.join(self.get_gencfg_path(), 'db'), log_prefix='diff_gencfg_db')
                have_diff = len(open(os.path.join(get_logs_folder(), 'diff_gencfg_db.out.txt')).read()) > 0

                nanny_diff_file = os.path.join(get_logs_folder(), 'nanny_diff')
                if have_diff:
                    run_process(['./tools/diffbuilder/main.py', '-u', '--nanny-filename', nanny_diff_file, '-d', 'group,tier,hw'], work_dir=self.get_gencfg_path(),
                                log_prefix='generate_smart_diff')

                    diff_size = os.stat(nanny_diff_file).st_size
                    if (not self.ctx[IgnoreDiffLimit.name]) and (diff_size > self.ctx[MaxDiffSize.name]):
                        raise Exception('Diff is too big. Actual size: {}. Max size: {}'.format(diff_size, self.ctx[MaxDiffSize.name]))

                    with open(os.path.join(get_logs_folder(), 'generate_smart_diff.out.txt')) as f:
                        self.ctx['diffbuilder_result'] = '\n'.join(next(f) for _ in xrange(300)) + \
                                                         '...\nSkipped. See task dir for full diffbuilder result.'

                    commit_msg = "[sandbox] [update_%s] Autoupdated by <%s> in sandbox task %s\n    Changes:\n%s" % (
                        self.ctx['update_entity'], updater.descr, self.http_url(),
                        '\n'.join(map(lambda x: "        %s" % x, self.ctx['changelog']))
                    )
                    commit_msg = commit_msg[:1000]

                    config_generator_utils.pull_gencfg_all(self.get_gencfg_path())
                    config_generator_utils.commit_gencfg_db(self.get_gencfg_path(), commit_msg)
                else:
                    self.ctx['status'] = EUpdaterStatus.NOTHINGTOCOMMIT
        finally:
            try:
                self._create_build_logs(self.get_gencfg_path())
            except Exception:
                pass


__Task__ = UpdateConfigGeneratorDb
