# -*- coding: utf-8 -*-
import os
import shutil
from datetime import datetime

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.common.rsync_helper import Rsync


class ExternalDataShare(SandboxStringParameter):
    name = "external_data_share"
    description = "External data share"
    group = "Userdata"
    default_value = "rsync://ya/berkanavt/database/dump/fasttier_catalog.tgz"


data_files = ["ua.query", "porno.query", "by.query", "kz.query", "clatrbase_config"]
index_files = ["geo.stat", "geoBase.c2CountryRegion", "geo.c2CountryCity",
               "cat.c2p", "genre.c2p", "geo.c2p", "onl.c2p", "sect.c2p", "src.c2p", "xxx.c2p", "geoa.c2p",
               "cat.c2s", "geo.c2s",
               "stopword.lst", "indexmisc", "indextp"]


data_dir_name = "fasttier_get_external_files"
index_dir_name = "indexfiles"


def download_data(share, destination):
    rsync = Rsync()
    process = rsync.get_file(share, destination)
    process.wait()
    return os.path.join(destination, os.path.basename(ExternalDataShare.default_value))


def get_timestamp():
    time = datetime.now()
    return time.strftime("%Y-%m-%d-%H-%M")


def patch_clatrbase_config(path):
    # Add extra attributes that Fasttier generates to the config. We don't want to modify common config so we patch it.
    with open(path, "a") as clatrbase:
        clatrbase.write("news.d2c:1\n")
        clatrbase.write("blogs.d2c:1\n")
        clatrbase.write("smi.d2c:1\n")
        clatrbase.write("turtwitter.d2c:8\n")


class UpdateExternalData(SandboxTask):
    """
    The task gets production configs for fusion
    """
    type = 'UPDATE_EXTERNAL_DATA'
    execution_space = 100 * 1024
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = (ExternalDataShare,)

    def arcadia_info(self):
        """Stub function to enable relases in sandbox"""
        return None, "Releasing Userdata", None

    def unpack_files(self, archive, files_list, tmp_dir):
        cmd = ["tar", "-vxzf", archive, "-C", tmp_dir] + files_list
        self._subprocess(cmd, wait=True, check=True, log_prefix="unpacking_data_files")
        return tmp_dir

    def on_execute(self):
        self.data_dir = self.path("%s.%s" % (data_dir_name, get_timestamp()))
        self.index_dir = os.path.join(self.data_dir, index_dir_name)
        self.tmp_dir = self.path("tmp_dir")

        for folder in [self.data_dir, self.tmp_dir, self.index_dir]:
            if not os.path.exists(folder):
                os.mkdir(folder)

        archive = download_data(self.ctx[ExternalDataShare.name], self.tmp_dir)
        tmp_dir = self.unpack_files(archive, index_files + data_files, self.tmp_dir)
        for file_path in os.listdir(tmp_dir):
            full_path = os.path.join(tmp_dir, file_path)

            if file_path in index_files:
                shutil.move(full_path, os.path.join(self.index_dir, file_path))

            if file_path in data_files:
                if file_path == "clatrbase_config":
                    patch_clatrbase_config(full_path)
                shutil.move(full_path, os.path.join(self.data_dir, file_path))

        self._create_resource(
            self.descr,
            self.data_dir,
            "FUSION_EXTERNAL_DATA",
            complete=True,
            attrs={"checked": True}
            )


__Task__ = UpdateExternalData
