# -*- coding: utf-8 -*-

from datetime import datetime

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.common.geosearch.update_env import UpdateGeoEnvironmentBase


class MapsDatabaseAdvertParameter(SandboxStringParameter):
    name = 'maps_database_advert_url'
    description = 'MAPS_DATABASE_ADVERT address: '
    required = False
    group = 'Update GeoMetaSearch Environment'


class MapsDatabaseBusinessParameter(SandboxStringParameter):
    name = 'maps_database_business_url'
    description = 'MAPS_DATABASE_BUSINESS address: '
    required = False
    group = 'Update GeoMetaSearch Environment'


class MapsDatabaseGeocoderParameter(SandboxStringParameter):
    name = 'maps_database_geocoder_url'
    description = 'MAPS_DATABASE_GEOCODER address: '
    required = False
    group = 'Update GeoMetaSearch Environment'


class MapsDatabaseWikicoderParameter(SandboxStringParameter):
    name = 'maps_database_wikicoder_url'
    description = 'MAPS_DATABASE_WIKICODER address: '
    required = False
    group = 'Update GeoMetaSearch Environment'


class MapsDatabaseWikiParameter(SandboxStringParameter):
    name = 'maps_database_wiki_url'
    description = 'MAPS_DATABASE_WIKI address: '
    required = False
    group = 'Update GeoMetaSearch Environment'


class UpdateGeoMetaSearchEnvironment(UpdateGeoEnvironmentBase):
    """
        Подготавливает окружение для тестирования геометапоиска
        репозиторием.
        Входы со scheduler:
        - список актуальных пакетов, которые надо ставить для всех запусков (MAPS_SEARCH_PACKAGES_LIST с атрибутом maps_search_packages_list)

        Названия всех атрибутов специально выбраны в стиле К.О.

        На выходе получаем:
        - Тот же MAPS_SEARCH_PACKAGES_LIST с атрибутом maps_search_packages_list_for_geometasearch
        - распакованные базы данных с названиями тегов + for_geometasearch

        У всех этих ресурсов значение тегов будет одинаковым
    """

    type = 'UPDATE_GEOMETASEARCH_ENVIRONMENT'

    required_ram = 50 << 10

    input_parameters = (
            MapsDatabaseAdvertParameter,
            MapsDatabaseBusinessParameter,
            MapsDatabaseGeocoderParameter,
            MapsDatabaseWikicoderParameter,
            MapsDatabaseWikiParameter,
        )

    client_tags = ctc.Tag.Group.LINUX
    cores = 17

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

    def on_execute(self):
        # Собираем входы
        pkg_list_res = self._get_res_with_attr('MAPS_SEARCH_PACKAGES_LIST', 'maps_search_packages_list_for_geometasearch')

        # Скачиваем
        res_list = [
            ('MAPS_DATABASE_ADVERT',   'maps_database_advert',    self.ctx.get(MapsDatabaseAdvertParameter.name)),
            ('MAPS_DATABASE_BUSINESS', 'maps_database_business',  self.ctx.get(MapsDatabaseBusinessParameter.name)),
            ('MAPS_DATABASE_GEOCODER', 'maps_database_geocoder',  self.ctx.get(MapsDatabaseGeocoderParameter.name)),
            ('MAPS_DATABASE_GEOCODER', 'maps_database_wikicoder', self.ctx.get(MapsDatabaseWikicoderParameter.name)),
            ('MAPS_DATABASE_WIKI',     'maps_database_wiki',      self.ctx.get(MapsDatabaseWikiParameter.name)),
        ]

        arch_list = []
        for _, _, url in res_list:
            if url:
                arch_path = self._download_arch(url)
            else:
                arch_path = ''
            arch_list.append(arch_path)

        # Распаковываем
        resources = []
        for (res_type, res_name, _), arch in zip(res_list, arch_list):
            descr = self.descr + ': ' + res_name
            if arch:
                res = self._unpack_to_resource(arch, res_name, res_type, resource_descr=descr)
            else:
                res = None
            resources.append(res)

        # Помечаем все выходные ресурсы timestamp
        mark = str(datetime.now())
        for (res_type, res_name, _), res in zip(res_list, resources):
            attr_name = res_name + '_for_geometasearch'
            if res:
                channel.sandbox.set_resource_attribute(res.id, attr_name, mark)

        # Помечаем ресурс с пакетами напоследок
        attr_name = 'maps_search_packages_list_for_geometasearch'
        channel.sandbox.set_resource_attribute(pkg_list_res.id, attr_name, mark)


__Task__ = UpdateGeoMetaSearchEnvironment
