# -*- coding: utf-8 -*-
import shutil
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxUrlParameter
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types

from sandbox.projects.common.utils import get_or_default
from sandbox.projects.common.geosearch.data_update import RSyncDataUpdater
from sandbox.projects.BuildMapsFilteredHostFactors import BuildMapsFilteredHostFactors

LAST_MODIFIED = 'last_modified'


def gen_source_url_param(index, value_pair):
    value, local_name = value_pair

    class DataSourceUrlParameter(SandboxUrlParameter):
        description = 'URL to download %s' % local_name
        name = 'data_source_url%d' % index
        file_name = local_name
        default_value = value
        required = True

    return DataSourceUrlParameter


sources = [('rsync://maps@lya.search.yandex.net/hostfactors/hostfactors.last.dat', 'hostfactors.dat'),
           ('rsync://maps@lya.search.yandex.net/hostfactors/hostfactors.last.idx', 'hostfactors.idx'),
           ('rsync://maps@lya.search.yandex.net/berkanavt/catalog/j-owners-spm.lst', 'j-owners-spm.lst'),
           ]


class UpdateMapsGeoHostFactors(SandboxTask):
    """
        Update host factors for geosearch business indexer
    """
    type = 'UPDATE_MAPS_GEO_HOST_FACTORS'

    input_parameters = [gen_source_url_param(index, value) for index, value in enumerate(sources)]

    def _get_res_descr(self, param):
        return {'url': get_or_default(self.ctx, param),
                'type': resource_types.OTHER_RESOURCE,
                'local_name': param.file_name,
                'no_dependency': True,
                }

    def on_execute(self):
        resources = {}
        for param in self.input_parameters:
            updater = RSyncDataUpdater(self)
            resources[param.name] = updater.update(self._get_res_descr(param), {'type': resource_types.MAPS_GEO_HOST_FACTORS})

        if not any(resources.itervalues()):
            return

        for param in self.input_parameters:
            if not resources[param.name]:
                updater = RSyncDataUpdater(self)
                resources[param.name] = updater.update(self._get_res_descr(param), None, True)

        last_modified = []
        for param in self.input_parameters:
            last_modified.append(channel.sandbox.get_resource_attribute(resources[param.name].id, LAST_MODIFIED))

        attributes = {}
        if last_modified:
            attributes[LAST_MODIFIED] = max(last_modified)

        outresource = self.create_resource('fresh_host_factors', 'hostfactors', resource_types.MAPS_GEO_HOST_FACTORS, attributes=attributes)
        outpath = outresource.path
        paths.make_folder(outpath)

        for param in self.input_parameters:
            paths.copy_path(resources[param.name].path, outpath, copy_function=shutil.move)

        for param in self.input_parameters:
            self.remove_resource_files(resources[param.name].id)
            channel.sandbox.delete_resource(resources[param.name].id)

        self.create_subtask(
            task_type=BuildMapsFilteredHostFactors.type,
            description='Filter fresh host factors',
            input_parameters={
                'host_factors': outresource.id,
            },
        )


__Task__ = UpdateMapsGeoHostFactors
