# -*- coding: utf-8 -*-

import os
import shutil
import urllib

from sandbox.common import errors
import sandbox.projects.BuildDockerImageV6 as bdiv6
from sandbox.sandboxsdk import process, parameters, environments


class UpdateRegularCoordinates(bdiv6.BuildDockerImageV6, object):
    """
    Пересчитывает регулярные координаты, загружает их на yt и затем,
    с помощью push-client, отправляет в logbroker для выгрузки в BigB.
    Если не указан Dockerfile url, то координаты не пересчитываются.
    Если не выбран UploadScriptBinaryResource или PushClientBinaryResource,
    то координаты не отправляются в logbroker.
    """

    type = "UPDATE_REGULAR_COORDINATES"
    max_restarts = 0

    # -------Resources for clustering:
    class TZDataResourceHttpUrl(parameters.SandboxStringParameter):
        name = 'tzdata_resource_http_url'
        default_value = 'https://proxy.sandbox.yandex-team.ru/last/GEODATATZDATA_STABLE'
        description = 'Geobase tzdata resource http url'
        required = True
        group = 'Resources for clustering'

    # -------Authentication inside docker:
    class VaultItemOwnerYt(parameters.SandboxStringParameter):
        name = 'vault_item_owner_yt'
        description = 'Vault item owner for yt token'
        default_value = 'robot-geolocation'
        group = 'Authentication inside docker'

    class VaultItemNameYt(parameters.SandboxStringParameter):
        name = 'vault_item_name_yt'
        description = (
            'Vault item with yt token for table creation (vault item name)'
        )
        default_value = 'robot-geolocation-yt-token'
        group = 'Authentication inside docker'

    class VaultItemOwnerSSH(parameters.SandboxStringParameter):
        name = 'vault_item_owner_ssh'
        description = 'Vault item owner for ssh private key'
        default_value = 'robot-geolocation'
        group = 'Authentication inside docker'

    class VaultItemNameSSH(parameters.SandboxStringParameter):
        name = 'vault_item_name_ssh'
        description = (
            'Vault item with ssh private key for usergeo clone (vault item name)'
        )
        default_value = 'robot-geolocation-ssh-key'
        group = 'Authentication inside docker'

    # -------Upload to BigB:
    class UploadScriptBinaryResource(parameters.ResourceSelector):
        name = "upload_script_binary_resource"
        description = "Arcadia binary resource with upload_regcoords_to_logbroker. " \
                      "Clear it to not upload regular coords to BigB."
        resource_type = []
        required = False
        group = 'Binary files for upload to logbroker'

    class PushClientBinaryResource(parameters.ResourceSelector):
        name = "push_client_binary_resource"
        description = "Push-client binary resource. " \
                      "Clear it to not upload regular coords to BigB."
        resource_type = []
        required = False
        group = 'Binary files for upload to logbroker'

    class YtTableName(parameters.SandboxStringParameter):
        name = 'yt_tabel_name'
        description = 'Input yt table name.'
        default_value = '//home/user_identification/usergeo/production/regular-371'
        group = 'Upload to BigB'

    class LogbrokerHostname(parameters.SandboxStringParameter):
        name = 'logbroker_hostname'
        description = 'Logbroker hostname'
        default_value = 'logbroker.yandex.net'
        group = 'Upload to BigB'

    class Ident(parameters.SandboxStringParameter):
        name = 'ident'
        description = 'Identifier'
        default_value = 'geolocation'
        group = 'Upload to BigB'

    class LogType(parameters.SandboxStringParameter):
        name = 'log_type'
        description = 'Log type'
        default_value = 'regular-geo-log'
        group = 'Upload to BigB'

    class YtJobDataSizeMegabytes(parameters.SandboxStringParameter):
        name = 'yt_job_data_size_megabytes'
        description = 'yt_job_data_size_megabytes'
        default_value = 20
        group = 'Upload to BigB'

    class YtJobUserSlots(parameters.SandboxStringParameter):
        name = 'yt_job_user_slots'
        description = 'yt_job_user_slots'
        default_value = 10
        group = 'Upload to BigB'

    input_parameters = [
        TZDataResourceHttpUrl,
        VaultItemOwnerYt,
        VaultItemNameYt,
        VaultItemOwnerSSH,
        VaultItemNameSSH,
        UploadScriptBinaryResource,
        PushClientBinaryResource,
        YtTableName,
        LogbrokerHostname,
        Ident,
        LogType,
        YtJobDataSizeMegabytes,
        YtJobUserSlots
    ] + bdiv6.BuildDockerImageV6.input_parameters

    def on_prepare(self):
        if self.dockerfile_url:
            bdiv6.BuildDockerImageV6.on_prepare(self)

            assert self.oauth_token, 'Oauth token must not be None or False'
            self._write_auth_resources()

            self._copy_tzdata_to_workdir()

        if self.upload_script_binary_resource and self.push_client_binary_resource:
            self._set_env_yt_token()

    def on_execute(self):
        if self.dockerfile_url:
            bdiv6.BuildDockerImageV6.prepare_docker_engine(self)
            self._docker_login()
            bdiv6.BuildDockerImageV6.build_docker_image(self)
            # Лучше всё же не давать даже возможность отправить собранный образ в registry
            # bdiv6.BuildDockerImageV6.docker_tag_and_push(self)
            self._run_docker_image()

        if self.upload_script_binary_resource and self.push_client_binary_resource:
            self._upload_to_bb()

    def on_finish(self):
        if self.dockerfile_url:
            bdiv6.BuildDockerImageV6.on_finish(self)
        self._remove_auth_resources()

    # Переопределяю метод, чтобы логиниться даже без указания registry tags
    def _docker_login(self):
        assert self.is_logged_in, 'You must be logged in with docker login'

    def _run_docker_image(self):
        process.run_process(list(self._docker_run_cmd),
                            shell=True, outputs_to_one_file=True, log_prefix='docker_run')

    def _set_env_yt_token(self):
        environments.SandboxEnvironment.update_os_env('YT_TOKEN', self._yt_token)

    def _upload_to_bb(self):
        process.run_process(list(self._upload_run_cmd),
                            shell=True, outputs_to_one_file=True, log_prefix='upload_run')

    @bdiv6.lazyprop
    def _docker_run_cmd(self):
        return self.docker_cmd + ('run', '-t', 'base')

    @bdiv6.lazyprop
    def _upload_run_cmd(self):
        return (
            self.upload_script_binary_resource_path,
            '--source-table', self.yt_tabel_name,
            '--push-client', self.push_client_binary_resource_path,
            '--logbroker-hostname', self.logbroker_hostname,
            '--ident', self.ident,
            '--log-type', self.log_type,
            '--volume', str(self.yt_job_data_size_megabytes),
            '--jobs', str(self.yt_job_user_slots),
            '--column', 'value'
        )

    @bdiv6.lazyprop
    def yt_token_owner(self):
        return self.ctx.get(self.VaultItemOwnerYt.name)

    @bdiv6.lazyprop
    def yt_token_name(self):
        return self.ctx.get(self.VaultItemNameYt.name)

    @bdiv6.lazyprop
    def git_key_owner(self):
        return self.ctx.get(self.VaultItemOwnerSSH.name)

    @bdiv6.lazyprop
    def git_key_name(self):
        return self.ctx.get(self.VaultItemNameSSH.name)

    @bdiv6.lazyprop
    def dockerfile_url(self):
        return self.ctx.get(bdiv6.BuildDockerImageV6.DockerfileUrl.name)

    @bdiv6.lazyprop
    def upload_script_binary_resource(self):
        return self.ctx[self.UploadScriptBinaryResource.name]

    @bdiv6.lazyprop
    def push_client_binary_resource(self):
        return self.ctx[self.PushClientBinaryResource.name]

    @bdiv6.lazyprop
    def upload_script_binary_resource_path(self):
        return self.sync_resource(self.upload_script_binary_resource)

    @bdiv6.lazyprop
    def push_client_binary_resource_path(self):
        return self.sync_resource(self.push_client_binary_resource)

    @bdiv6.lazyprop
    def yt_tabel_name(self):
        return self.ctx.get(self.YtTableName.name)

    @bdiv6.lazyprop
    def logbroker_hostname(self):
        return self.ctx.get(self.LogbrokerHostname.name)

    @bdiv6.lazyprop
    def ident(self):
        return self.ctx.get(self.Ident.name)

    @bdiv6.lazyprop
    def log_type(self):
        return self.ctx.get(self.LogType.name)

    @bdiv6.lazyprop
    def yt_job_data_size_megabytes(self):
        return self.ctx.get(self.YtJobDataSizeMegabytes.name)

    @bdiv6.lazyprop
    def yt_job_user_slots(self):
        return self.ctx.get(self.YtJobUserSlots.name)

    @bdiv6.lazyprop
    def yt_token_path(self):
        return os.path.join(os.path.join(self.res_path, ".yt"), "token")

    @bdiv6.lazyprop
    def git_key_path(self):
        return os.path.join(os.path.join(self.res_path, ".ssh"), "robot_rsa")

    @bdiv6.lazyprop
    def _yt_token(self):
        if self.yt_token_owner and self.yt_token_name:
            return self.get_vault_data(self.yt_token_owner, self.yt_token_name)
        elif self.token_name:
            return self.get_vault_data(self.yt_token_name)
        else:
            return None

    @bdiv6.lazyprop
    def _git_ssh_key(self):
        if self.git_key_owner and self.git_key_name:
            return self.get_vault_data(self.git_key_owner, self.git_key_name)
        elif self.git_key_name:
            return self.get_vault_data(self.git_key_name)
        else:
            return None

    @staticmethod
    def assure_path_exists(path):
        directory = os.path.dirname(path)
        if not os.path.exists(directory):
            os.makedirs(directory)

    @staticmethod
    def write_safe_res(path, content):
        assert content, "an oauth resource must not be None or False"
        UpdateRegularCoordinates.assure_path_exists(path)
        with open(path, "a") as out_file:
            out_file.write(content)
        directory = os.path.dirname(path)
        os.chmod(directory, 0700)
        os.chmod(path, 0400)

    @staticmethod
    def safe_remove(file_path):
        try:
            os.remove(file_path)
        except OSError as e:
            if e.errno != 2:
                raise errors.TaskFailure(e)

    def _write_auth_resources(self):
        """записывает yt_token_path и git_key_path"""
        self.write_safe_res(self.git_key_path, self._git_ssh_key)
        self.write_safe_res(self.yt_token_path, self._yt_token)

    def _remove_auth_resources(self):
        """удаляет yt_token_path и git_key_path, если они существуют"""
        self.safe_remove(self.yt_token_path)
        self.safe_remove(self.git_key_path)

    @staticmethod
    def find(name, path):
        for root, dirs, files in os.walk(path):
            if name in files:
                return os.path.join(root, name)

    def _copy_tzdata_to_workdir(self):
        file_name = "tzdata.tar.gz"
        tzdata_file_path = os.path.join(self.res_path, file_name)

        urllib.urlretrieve(self.ctx[self.TZDataResourceHttpUrl.name], tzdata_file_path)

        if os.path.isdir(tzdata_file_path):
            tzdata_dir_name = os.path.join(self.res_path, "tzdata")
            os.rename(tzdata_file_path, tzdata_dir_name)

            file_path = self.find(file_name, tzdata_dir_name)
            assert file_path, "Can't find %s in %s." % (file_name, tzdata_dir_name)

            shutil.copyfile(file_path, tzdata_file_path)
        else:
            assert os.path.isfile(tzdata_file_path), "File %s doesn't exist." % tzdata_file_path


__Task__ = UpdateRegularCoordinates
