# -*- coding: utf-8 -*-

import os

from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter

from sandbox.projects.common.geosearch.update_env import UpdateGeoEnvironmentBase


def create_param_resource_from_svn(
    param_name=None,
    param_description=None,
    param_group=None
):
    class RevisionParameter(SandboxIntegerParameter):
        name = param_name
        description = param_description + ', revision'
        group = param_group
        required = False

    return RevisionParameter


SVN_ARCADIA_ROOT_PATH = 'arcadia:/arc/trunk/arcadia_tests_data/geosearch/sandbox_test_data/'
SVN_ARCADIA_PACKAGES_ROOT = 'arcadia:/arc/trunk/arcadia/packages/debian/geosearch/'


def _cut_comments_from_queries(file_path):
    with open(file_path, 'r') as f:
        lines = f.readlines()

    with open(file_path, 'w') as f:
        for l in lines:
            if l.startswith('#'):
                continue
            if l.isspace():
                continue
            f.write(l)


class ResourceDescription(object):
    arcadia_path = None
    resource_type = None
    attribute = None
    description = None

    @staticmethod
    def postprocess(file_path):
        pass


class QueriesForGeometesearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'geometasearch/queries.txt'
    resource_type = 'PLAIN_TEXT_QUERIES'
    attribute = 'queries_for_geometasearch'
    description = 'Queries for upper geometasearch'

    param = create_param_resource_from_svn(
        param_name='geoupper_queries',
        param_description=description,
        param_group='Geometasearch tests',
    )

    @staticmethod
    def postprocess(file_path):
        _cut_comments_from_queries(file_path)


class ConfigForUpperGeometesearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'geometasearch/sandboxhost--8031.cfg'
    resource_type = 'GEOMETASEARCH_CONFIG'
    attribute = 'config_for_geometasearch_upper'
    description = 'Config for upper geometasearch'

    param = create_param_resource_from_svn(
        param_name='config_for_geometasearch_upper',
        param_description=description,
        param_group='Geometasearch tests',
    )

    @staticmethod
    def postprocess(file_path):
        _cut_comments_from_queries(file_path)


class ConfigForMiddleGeometesearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'geometasearch/sandboxhost--8032.cfg'
    resource_type = 'GEOMETASEARCH_CONFIG'
    attribute = 'config_for_geometasearch_middle'
    description = 'Config for middle geometasearch'

    param = create_param_resource_from_svn(
        param_name='config_for_geometasearch_middle',
        param_description=description,
        param_group='Geometasearch tests',
    )

    @staticmethod
    def postprocess(file_path):
        _cut_comments_from_queries(file_path)


class BusinessConfigForGeometasearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_PACKAGES_ROOT + 'yandex-geosearch-basesearch-all/etc/yandex/maps/search/business/business.cfg'
    resource_type = 'GEOBASESEARCH_CONFIG'
    attribute = 'business_config_geobasesearch_for_geometasearch'
    description = 'business.cfg for geosearch-business'

    param = create_param_resource_from_svn(
        param_name='business_geobasesearch_cfg',
        param_description=description,
        param_group='Geometasearch tests',
    )


class GeocoderConfigGeobasesearchForGeometasearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'geometasearch/geocoder_geobasesearch.cfg'
    resource_type = 'GEOBASESEARCH_CONFIG'
    attribute = 'geocoder_config_yserver_for_geometasearch'
    description = 'geobasesearch.cfg for maps-search-geocoder'

    param = create_param_resource_from_svn(
        param_name='geocoder_geobasesearch_cfg',
        param_description=description,
        param_group='Geometasearch tests',
    )


class GeocoderConfigForGeometasearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'geometasearch/geocoder_geocoder.cfg'
    resource_type = 'MAPS_SEARCH_CONFIG_GEOCODER'
    attribute = 'geocoder_config_for_geometasearch'
    description = 'geocoder.cfg for maps-search-geocoder'

    param = create_param_resource_from_svn(
        param_name='geocoder_cfg',
        param_description=description,
        param_group='Geometasearch tests',
    )


class WikiConfigForGeometasearch(ResourceDescription):
    arcadia_path = SVN_ARCADIA_PACKAGES_ROOT + 'yandex-geosearch-basesearch-all/etc/yandex/maps/search/wiki/wiki.cfg'

    resource_type = 'GEOBASESEARCH_CONFIG'
    attribute = 'wiki_config_geobasesearch_for_geometasearch'
    description = 'wiki.cfg for maps-search-wiki'

    param = create_param_resource_from_svn(
        param_name='wiki_basesearch_cfg',
        param_description=description,
        param_group='Geometasearch tests',
    )


#
# Maps tests
#

class QueriesForBusiness(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'maps-search-business/queries.txt'
    resource_type = 'PLAIN_TEXT_QUERIES'
    attribute = 'queries_for_maps_search_business'
    description = 'Queries for maps-search-business'

    param = create_param_resource_from_svn(
        param_name='business_queries',
        param_description=description,
        param_group='Maps-search-business tests',
    )

    @staticmethod
    def postprocess(file_path):
        _cut_comments_from_queries(file_path)


class ConfigGeobasesearchForBusiness(ResourceDescription):
    arcadia_path = SVN_ARCADIA_PACKAGES_ROOT + 'yandex-geosearch-basesearch-all/etc/yandex/maps/search/business/business.cfg'

    resource_type = 'GEOBASESEARCH_CONFIG'
    attribute = 'config_geobasesearch_for_maps_search_business'
    description = 'business.cfg for maps-search-business'

    param = create_param_resource_from_svn(
        param_name='geobasesearch_cfg',
        param_description=description,
        param_group='Maps-search-business tests',
    )


class SmallAdvertForBusiness(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'maps-search-business/small_advert.bin'
    resource_type = 'MAPS_DATABASE_ADVERT'
    attribute = 'small_advert_for_maps_search_business'
    description = 'Small advert.bin for maps-search-business'

    param = create_param_resource_from_svn(
        param_name='business_small_advert',
        param_description=description,
        param_group='Maps-search-business tests',
    )


class SmallBackaForBusiness(ResourceDescription):
    arcadia_path = SVN_ARCADIA_ROOT_PATH + 'maps-search-business/small_backa'
    resource_type = 'MAPS_DATABASE_BUSINESS_SOURCE'
    attribute = 'small_backa_for_maps_search_business'
    description = 'Small backa for maps-search-business'

    param = create_param_resource_from_svn(
        param_name='business_small_backa',
        param_description=description,
        param_group='Maps-search-business tests',
    )


RESOURCE_DESCRS = (
    QueriesForGeometesearch,
    ConfigForUpperGeometesearch,
    ConfigForMiddleGeometesearch,

    BusinessConfigForGeometasearch,

    GeocoderConfigGeobasesearchForGeometasearch,
    GeocoderConfigForGeometasearch,

    WikiConfigForGeometasearch,

    QueriesForBusiness,
    ConfigGeobasesearchForBusiness,
    SmallAdvertForBusiness,
    SmallBackaForBusiness,
)


class UpdateSvnResoucesForGeoTests(UpdateGeoEnvironmentBase):
    """
        Выкачивает файлы из svn и создает из них ресурсы,
        которые должен подхватить test_environment
    """

    type = 'UPDATE_SVN_RESOUCES_FOR_GEO_TESTS'

    input_parameters = [res.param for res in RESOURCE_DESCRS]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self

    def on_execute(self):
        for res_descr in RESOURCE_DESCRS:
            revision = self.ctx.get(res_descr.param.name)
            if not revision:
                continue

            res_filename = os.path.basename(res_descr.arcadia_path)
            res_path = self.path(res_filename)
            svn_url = res_descr.arcadia_path + '@' + str(revision)
            Arcadia.export(svn_url, res_path)

            res_descr.postprocess(res_path)

            args = (res_descr.description, res_path, res_descr.resource_type)
            kwargs = {
                'attributes': {
                    res_descr.attribute: 'yes',
                    'resource_revision': str(revision),
                }
            }

            self.create_resource(*args, **kwargs)


__Task__ = UpdateSvnResoucesForGeoTests
